#include "decoder.h"
#include "graphite.h"

#include <solomon/agent/lib/storage/sharded_storage.h>
#include <solomon/agent/modules/agent/graphite/protos/graphite_config.pb.h>
#include <solomon/agent/modules/agent/graphite/tcp/tcp.h>
#include <solomon/agent/misc/logger.h>

#include <util/memory/blob.h>

using namespace NMonitoring;

namespace NSolomon {
namespace NAgent {

struct TShardId {
    TString Project;
    TString Service;
};

class TCallback: public TTcpServer::ICallback {
public:
    TCallback(const TGraphiteConversionConfig& config, IShardConsumerProvider* writer, TShardId shardId)
        : Decoder_{CreateGraphiteDecoder(config)}
        , Writer_{writer}
        , ShardId_{std::move(shardId)}
    {
    }

    TTcpServer::TConnectionHandler* CreateHandler() override;

private:
    IDecoderPtr Decoder_;
    IShardConsumerProvider* Writer_;
    TShardId ShardId_;
};

class TGraphiteHandler: public TTcpServer::TConnectionHandler {
public:
    TGraphiteHandler(IDecoder& decoder, IStorageMetricsConsumerPtr consumer)
        : Consumer_{std::move(consumer)}
        , GraphiteDecoder_{decoder}
    {
    }

    bool Decode(TStringBuf data) noexcept try {
        GraphiteDecoder_.Decode(data, *Consumer_);
        return true;
    } catch (...) {
        Cerr << "Failed to decode data: " << CurrentExceptionMessage();
        return false;
    }

    void OnConnectionOpen(IConnection* conn) noexcept override {
        Start_ = TInstant::Now();
        Conn_ = conn;
    }

    void Flush() noexcept try {
        Consumer_->Flush();
    } catch (...) {
        SA_LOG(ERROR) << "Writing to storage from graphite module failed: " << CurrentExceptionMessage();
    }

    void OnConnectionClose(const TBlob& remainingData) noexcept override {
        if (Decode({remainingData.AsCharPtr(), remainingData.Size()})) {
            Flush();
        }
    }

    size_t OnData(const TBlob& data) noexcept override {
        auto d = TStringBuf{data.AsCharPtr(), data.Size()};
        auto lf = d.rfind('\n');
        if (lf == TStringBuf::npos) {
            return 0;
        }

        d.Trunc(lf + 1);
        if (!Decode(d)) {
            Conn_->Close();
            return data.Size();
        }

        return d.size();
    }

private:
    IConnection* Conn_;
    TInstant Start_;
    IStorageMetricsConsumerPtr Consumer_;
    IDecoder& GraphiteDecoder_;
};

TTcpServer::TConnectionHandler* TCallback::CreateHandler() {
    return new TGraphiteHandler(*Decoder_, Writer_->CreateShardConsumer(ShardId_.Project, ShardId_.Service));
}

class TGraphiteModule : public IAgentPushModule {
public:
    ~TGraphiteModule() {
        Stop();
    }

    TGraphiteModule(
        const TGraphitePushModuleConfig& config,
        IShardConsumerProvider* writer,
        class ITcpServerStatusListener* listener
    )
        : Name_{config.GetName()}
    {
        Y_ENSURE(config.HasMappingRules(), "Mapping rules are required");
        TTcpServer::TOptions opts;
        if (config.GetThreadCount() != 0) {
            opts.ThreadCount = config.GetThreadCount();
        }

        Y_ENSURE(!config.GetBindAddress().empty() && config.GetBindPort() != 0, "Port and address are required");

        const auto& project = config.GetProject();
        const auto& service = config.GetService();

        Y_ENSURE(!project.empty() && !service.empty());
        opts.BindAddress = config.GetBindAddress();
        opts.BindPort = config.GetBindPort();

        Srv_.Reset(new TTcpServer{::MakeIntrusive<TCallback>(
                config.GetMappingRules(),
                writer,
                TShardId{project, service}),
            opts, listener});
    }

    void Start() override {
        Srv_->Start();
    }

    void Stop() override {
        Srv_->Stop();
    }

    TStringBuf Name() const override {
        return Name_;
    }

private:
    THolder<TTcpServer> Srv_;
    const TString Name_;
};

TAgentPushModulePtr CreateGraphitePushModule(
    const TGraphitePushModuleConfig& config,
    IShardConsumerProvider* writer,
    class ITcpServerStatusListener* listener)
{
    return ::MakeHolder<TGraphiteModule>(config, writer, listener);
}


} // namespace NAgent
} // namespace NSolomon
