#include <solomon/agent/modules/agent/graphite/tcp/tcp.h>
#include <solomon/agent/modules/agent/graphite/tcp/data_stream.h>

#include <library/cpp/testing/common/network.h>
#include <library/cpp/testing/gtest/gtest.h>

#include <util/stream/str.h>
#include <util/stream/mem.h>
#include <util/system/event.h>
#include <util/network/socket.h>
#include <util/memory/blob.h>

using namespace NSolomon::NAgent;

class TTestHandler : public TTcpServer::TConnectionHandler {
public:
    TTestHandler(TStringStream& os, TStringStream& remaining, TAutoEvent& done)
        : Stream{os}
        , Remaining{remaining}
        , Done{done}
    {
    }

    size_t OnData(const TBlob& data) noexcept override {
        auto d = TStringBuf{data.AsCharPtr(), data.Size()};
        auto lf = d.rfind('\n');
        if (lf != TStringBuf::npos) {
            d.Trunc(lf + 1);
        }

        Stream << d;
        Stream.Flush();

        return d.size();
    }

    void OnConnectionClose(const TBlob& data) noexcept override {
        Remaining << TStringBuf{data.AsCharPtr(), data.Size()};
        Stream.Flush();

        Done.Signal();
    }

    TStringStream& Stream;
    TStringStream& Remaining;
    TAutoEvent& Done;
};

class TTcpTest : public TTcpServer::ICallback {
public:
    TTcpServer::TConnectionHandler* CreateHandler() override {
        return new TTestHandler{Received, Remaining, Done};
    }

public:
    TStringStream Received;
    TStringStream Remaining;

    TAutoEvent Done;
};

TEST(TTcpServerTest, ServerWorks) {
    auto port = NTesting::GetFreePort();
    auto callback = MakeIntrusive<TTcpTest>();

    TTcpServer::TOptions options;
    options.BindAddress = "localhost";
    options.BindPort = port;
    options.BufferSize = 20;

    TTcpServer server{callback, options};
    server.Start();

    TString request;

    {
        TSocket socket({"localhost", port});
        TSocketOutput output(socket);

        request = "foo\nbar\nbaz\nfizz\nbuzz";
        output << request;
    }

    callback->Done.WaitT(TDuration::MilliSeconds(50));

    TString actual = callback->Received.Str();
    ASSERT_EQ("foo\nbar\nbaz\nfizz\n", actual);

    TString remaining = callback->Remaining.Str();
    ASSERT_EQ("buzz", remaining);
    server.Stop();
}

TEST(TDataStreamTest, DataStream)  {
    TDataStream ds{8};

    auto write = [&] (TString what) {
        TMemoryOutput out{ds.ReadBuf(), ds.Avail()};
        out << what;
        ds.MarkWritten(what.size());
    };

    auto asBuf = [] (auto&& ds) {
        auto&& blob = ds.AsBlob();
        return TStringBuf{blob.AsCharPtr(), blob.Size()};
    };

    write("ab");
    ASSERT_EQ(asBuf(ds), "ab");

    write("cdefgh");
    ASSERT_EQ(asBuf(ds), "abcdefgh");
    ASSERT_EQ(ds.Avail(), 0u);
    ASSERT_TRUE(ds.IsFull());

    ds.MarkRead(0);
    ASSERT_EQ(asBuf(ds), "abcdefgh");

    ds.MarkRead(2);
    ASSERT_EQ(asBuf(ds), "cdefgh");

    ds.MarkRead(1);
    ASSERT_EQ(asBuf(ds), "defgh");
    ASSERT_EQ(ds.Avail(), 3u);
    ASSERT_FALSE(ds.IsFull());
}
