#include <solomon/agent/modules/agent/graphite/decoder.h>

#include <library/cpp/testing/gtest/gtest.h>

#include <solomon/agent/modules/agent/graphite/protos/graphite_config.pb.h>

#include <library/cpp/monlib/metrics/labels.h>
#include <library/cpp/monlib/encode/protobuf/protobuf.h>

#include <google/protobuf/arena.h>
#include <google/protobuf/text_format.h>

using namespace NSolomon::NAgent;
using namespace NMonitoring;


IDecoderPtr CreateDecoder(TGraphiteConversionConfig::EMode mode, TVector<TString> blacklist = {}) {
    TGraphiteConversionConfig config;
    config.SetMode(mode);
    auto* mapping = config.AddMetrics();

    mapping->SetPattern("*.foo");

    auto* label = mapping->AddLabelMappings();
    label->SetLabel("notMetric");
    label->SetTemplate("$1");

    TGraphiteMetricBlacklist list;
    for (auto&& pattern: blacklist) {
        *list.AddPattern() = pattern;
    }
    config.MutableBlacklist()->CopyFrom(list);

    return CreateGraphiteDecoder(config);
}

TEST(TGraphiteConverter, Simple) {
    TGraphiteConversionConfig config;
    auto* mapping = config.AddMetrics();
    mapping->SetPattern("*.foo");

    auto* label = mapping->AddLabelMappings();
    label->SetLabel("metric");
    label->SetTemplate("$1");
    auto map = CreateGraphiteLabelConverter(config);

    const auto labels = map->Convert("bar.foo").GetRef();
    ASSERT_EQ(labels.Size(), 1u);
    ASSERT_EQ(labels[0].Name(), "metric");
    ASSERT_EQ(labels[0].Value(), "bar");
}

TEST(TGraphiteConverter, Regression1) {
    const auto confText = R"(Mode: IGNORE
Metrics {
    Pattern: "direct_one_min.db_configurations.*.flow.camp_auto_price.*.*.*"
    LabelMappings {
        Label: "entity"
        Template: "camp_auto_price"
    }
    LabelMappings {
        Label: "env"
        Template: "$1"
    }
    LabelMappings {
        Label: "queue"
        Template: "$2"
    }
    LabelMappings {
        Label: "metric"
        Template: "$3"
    }
    LabelMappings {
        Label: "shard"
        Template: "$4"
    }
}

Metrics {
    Pattern: "direct_one_min.db_configurations.*.flow.camp_auto_price.*.queue_size.*.0"
    LabelMappings {
        Label: "entity"
        Template: "camp_auto_price"
    }
    LabelMappings {
        Label: "metric"
        Template: "queue_size"
    }
    LabelMappings {
        Label: "env"
        Template: "$1"
    }
    LabelMappings {
        Label: "queue"
        Template: "$2"
    }
    LabelMappings {
        Label: "shard"
        Template: "$3"
    }
})";

    TGraphiteConversionConfig config;
    NProtoBuf::TextFormat::ParseFromString(confText, &config);

    auto map = CreateGraphiteLabelConverter(config);
    const auto labels = map->Convert("direct_one_min.db_configurations.devtest.flow.camp_auto_price.easy1.queue_size.shard_2.0").GetRef();

    ASSERT_EQ(labels.Size(), 5u);
    THashMap<TString, TString> result;
    for (const auto& l: labels) {
        result[l.Name()] = l.Value();
    }

    ASSERT_EQ(result.at("metric"), "queue_size");
    ASSERT_EQ(result.at("entity"), "camp_auto_price");
    ASSERT_EQ(result.at("queue"), "easy1");
    ASSERT_EQ(result.at("env"), "devtest");
    ASSERT_EQ(result.at("shard"), "shard_2");
}

TEST(TGraphiteConverter, Templating) {
    TGraphiteConversionConfig config;
    auto* mapping = config.AddMetrics();
    mapping->SetPattern("*.*.*");

    auto* label = mapping->AddLabelMappings();
    label->SetLabel("metric");
    label->SetTemplate("$1");

    label = mapping->AddLabelMappings();
    label->SetLabel("label2");
    label->SetTemplate("$2_foo");

    label = mapping->AddLabelMappings();
    label->SetLabel("label3");
    label->SetTemplate("bar_$3");

    label = mapping->AddLabelMappings();
    label->SetLabel("label4");
    label->SetTemplate("fizz");

    label = mapping->AddLabelMappings();
    label->SetLabel("label5");
    label->SetTemplate("$1_$2");

    auto map = CreateGraphiteLabelConverter(config);
    const auto labels = map->Convert("foo.bar.baz").GetRef();

    ASSERT_EQ(labels.Size(), 5u);
    THashMap<TString, TString> result;
    for (const auto& l: labels) {
        result[l.Name()] = l.Value();
    }

    ASSERT_EQ(result.at("metric"), "foo");
    ASSERT_EQ(result.at("label2"), "bar_foo");
    ASSERT_EQ(result.at("label3"), "bar_baz");
    ASSERT_EQ(result.at("label4"), "fizz");
    ASSERT_EQ(result.at("label5"), "foo_bar");
}

TEST(TGraphiteConverter, NonExistingReference) {
    TGraphiteConversionConfig config;
    auto* mapping = config.AddMetrics();
    mapping->SetPattern("*.foo");

    auto* label = mapping->AddLabelMappings();
    label->SetLabel("metric");
    label->SetTemplate("$2");

     ASSERT_THROW(CreateGraphiteLabelConverter(config), yexception);
}

TEST(TGraphiteConverter, NoWildcardsPattern) {
    TGraphiteConversionConfig config;

    auto* mapping = config.AddMetrics();
    mapping->SetPattern("fooBar");

    auto* label = mapping->AddLabelMappings();
    label->SetLabel("metricName");
    label->SetTemplate("fooBarBaz");

    auto map = CreateGraphiteLabelConverter(config);
    const auto labels = map->Convert("fooBar").GetRef();

    ASSERT_EQ(labels.Size(), 1u);
    ASSERT_EQ(labels[0].Name(), "metricName");
    ASSERT_EQ(labels[0].Value(), "fooBarBaz");

}

TEST(TGraphiteConverter, ConversionWithSpecialChars) {
    TGraphiteConversionConfig config;

    auto* mapping = config.AddMetrics();
    mapping->SetPattern("*.foo.*");

    auto* label = mapping->AddLabelMappings();
    label->SetLabel("metric");
    label->SetTemplate("$1-2");

    label = mapping->AddLabelMappings();
    label->SetLabel("label2");
    label->SetTemplate("$2_foo");

    auto map = CreateGraphiteLabelConverter(config);
    const auto labels = map->Convert("prefix2.foo.suffix-3").GetRef();

    ASSERT_EQ(labels.Size(), 2u);
    THashMap<TString, TString> result;
    for (const auto& l: labels) {
        result[l.Name()] = l.Value();
    }

    ASSERT_EQ(result.at("metric"), "prefix2-2");
    ASSERT_EQ(result.at("label2"), "suffix-3_foo");
}

TEST(TGraphiteConverter, StrictPolicy) {
    NProtoBuf::Arena arena;
    auto samples = NProtoBuf::Arena::CreateMessage<NProto::TMultiSamplesList>(&arena);

    auto encoder = EncoderProtobuf(samples);
    auto decoder = CreateDecoder(TGraphiteConversionConfig::IGNORE);

    decoder->Decode(TStringBuf{"bar.baz.fiz 0 0"}, *encoder);
    ASSERT_EQ(samples->SamplesSize(), 0u);
}

TEST(TGraphiteConverter, AsIsPolicy) {
    NProtoBuf::Arena arena;
    auto samples = NProtoBuf::Arena::CreateMessage<NProto::TMultiSamplesList>(&arena);

    auto encoder = EncoderProtobuf(samples);
    auto decoder = CreateDecoder(TGraphiteConversionConfig::AS_IS);

    decoder->Decode(TStringBuf{"bar.baz.fiz 0 0"}, *encoder);

    ASSERT_EQ(samples->SamplesSize(), 1u);
    const auto& s = samples->GetSamples(0);
    ASSERT_EQ(s.GetMetricType(), NProto::GAUGE);

    ASSERT_EQ(s.LabelsSize(), 1u);
    ASSERT_EQ(s.GetLabels(0).GetName(), "sensor");
    ASSERT_EQ(s.GetLabels(0).GetValue(), "bar.baz.fiz");
}

TEST(TGraphiteConverter, DecodeNormal) {
    NProtoBuf::Arena arena;
    auto samples = NProtoBuf::Arena::CreateMessage<NProto::TMultiSamplesList>(&arena);

    auto encoder = EncoderProtobuf(samples);
    auto decoder = CreateDecoder(TGraphiteConversionConfig::IGNORE);

    decoder->Decode(TStringBuf{"bar.foo 0 0"}, *encoder);

    ASSERT_EQ(samples->SamplesSize(), 1u);
    const auto& s = samples->GetSamples(0);
    ASSERT_EQ(s.GetMetricType(), NProto::GAUGE);

    ASSERT_EQ(s.LabelsSize(), 1u);
    ASSERT_EQ(s.GetLabels(0).GetName(), "notMetric");
    ASSERT_EQ(s.GetLabels(0).GetValue(), "bar");
}

TEST(TGraphiteConverter, SimpleBlacklist) {
    NProtoBuf::Arena arena;
    TVector<TString> blacklist {
        {"*.foo"}
    };

    auto samples = NProtoBuf::Arena::CreateMessage<NProto::TMultiSamplesList>(&arena);

    auto encoder = EncoderProtobuf(samples);
    auto decoder = CreateDecoder(TGraphiteConversionConfig::AS_IS, blacklist);

    decoder->Decode(TStringBuf{"bar.foo 0 0\nfoo.bar 0 42"}, *encoder);

    ASSERT_EQ(samples->SamplesSize(), 1u);
    const auto& s = samples->GetSamples(0);
    ASSERT_EQ(s.GetMetricType(), NProto::GAUGE);

    ASSERT_EQ(s.LabelsSize(), 1u);
    ASSERT_EQ(s.GetLabels(0).GetName(), "sensor");
    ASSERT_EQ(s.GetLabels(0).GetValue(), "foo.bar");
}

TEST(TGraphiteConverter, MultipleBlacklist) {
    NProtoBuf::Arena arena;
    TVector<TString> blacklist {
        {"*.foo"},
        {"foo.bar"},
        {"*.baz"},
    };

    auto samples = NProtoBuf::Arena::CreateMessage<NProto::TMultiSamplesList>(&arena);

    auto encoder = EncoderProtobuf(samples);
    auto decoder = CreateDecoder(TGraphiteConversionConfig::AS_IS, blacklist);

    decoder->Decode(TStringBuf{"bar.foo 0 0\nfoo.bar 0 42\nbaz 1 2"}, *encoder);

    ASSERT_EQ(samples->SamplesSize(), 1u);
    const auto& s = samples->GetSamples(0);
    ASSERT_EQ(s.GetMetricType(), NProto::GAUGE);

    ASSERT_EQ(s.LabelsSize(), 1u);
    ASSERT_EQ(s.GetLabels(0).GetName(), "sensor");
    ASSERT_EQ(s.GetLabels(0).GetValue(), "baz");
}

TEST(TGraphiteConverter, InvalidBlacklist) {
    NProtoBuf::Arena arena;
    TVector<TString> blacklist {
        {"[a-z]+.foo"},
    };

    auto samples = NProtoBuf::Arena::CreateMessage<NProto::TMultiSamplesList>(&arena);

    auto encoder = EncoderProtobuf(samples);
    ASSERT_THROW(CreateDecoder(TGraphiteConversionConfig::AS_IS, blacklist), yexception);
}
