#include "push.h"

#include <solomon/agent/lib/http/handler.h>
#include <solomon/agent/lib/http/server.h>
#include <solomon/agent/lib/storage/sharded_storage.h>

#include <solomon/agent/protos/http_server_config.pb.h>
#include <solomon/agent/modules/agent/push/protos/push_config.pb.h>

#include <library/cpp/http/misc/httpcodes.h>

#include <library/cpp/monlib/encode/format.h>

#include <library/cpp/monlib/encode/json/json.h>
#include <library/cpp/monlib/encode/spack/spack_v1.h>

#include <util/stream/mem.h>

using namespace NMonitoring;

namespace NSolomon {
namespace NAgent {

namespace {

TString ModuleNameFromConfig(const THttpPushModuleConfig& config) {
    TString name = config.GetName();
    if (!name.empty()) {
        return name;
    }

    return TStringBuilder() << "HttpPush/" << config.GetBindAddress() << ":" << config.GetBindPort();
}

class TSolomonPushHandler : public IHttpHandler {
public:
    TSolomonPushHandler(TString project, TString service, IShardConsumerProvider* writer)
        : Writer_{writer}
        , Project_{std::move(project)}
        , Service_{std::move(service)}
    {
    }

    void OnPost(const THttpRequest& req, THttpResponse* resp) override {
        EFormat result{EFormat::UNKNOWN};

        if (TStringBuf contentType = req.RD.HeaderInOrEmpty("content-type")) {
            result = FormatFromContentType(contentType);
        }

        IStorageMetricsConsumerPtr consumer = Writer_->CreateShardConsumer(Project_, Service_);

        if (result == EFormat::JSON) {
            DecodeJson({req.Body.AsCharPtr(), req.Body.Size()}, consumer.Get());
        } else if (result == EFormat::SPACK) {
            TMemoryInput memIn{req.Body.AsCharPtr(), req.Body.Size()};
            DecodeSpackV1(&memIn, consumer.Get());
        } else {
            ythrow THttpError(HTTP_BAD_REQUEST)
                << "content-type must be one of application/json, application/x-solomon-spack";
        }

        consumer->Flush();

        resp->SetHttpCode(HTTP_OK);
    }

private:
    IShardConsumerProvider* Writer_;
    TString Project_;
    TString Service_;
};

} // namespace


class THttpPushModule : public IAgentPushModule {
public:
    THttpPushModule(
        const THttpPushModuleConfig& config,
        IThreadPoolProvider& threadPoolProvider,
        IShardConsumerProvider* writer,
        IServerStatusListener* listener);

    TStringBuf Name() const override {
        return Name_;
    }

    void Start() override;
    void Stop() override;

private:
    THolder<NSolomon::NAgent::THttpServer> Server_;
    const TString Name_;
};

THttpPushModule::THttpPushModule(
        const THttpPushModuleConfig& config,
        IThreadPoolProvider& threadPoolProvider,
        IShardConsumerProvider* writer,
        IServerStatusListener* listener)
    : Name_{ModuleNameFromConfig(config)}
{
    Y_ENSURE(config.GetBindPort() != 0 && !config.GetBindAddress().empty(),
        "Bind address and port must be specified for a push module");

    THttpServerConfig serverConfig;
    serverConfig.SetBindAddress(config.GetBindAddress());
    serverConfig.SetBindPort(config.GetBindPort());

    const TString& poolName = config.GetThreadPoolName();
    if (poolName)
        serverConfig.SetThreadPoolName(poolName);

    Y_VERIFY(config.GetBindPort() <= std::numeric_limits<ui16>::max());

    Server_.Reset(new THttpServer{serverConfig, threadPoolProvider, listener});

    for (const auto& handlerConfig: config.GetHandlers()) {
        Y_ENSURE(handlerConfig.GetFormat() == TPushHandlerConfig::SOLOMON,
            "Only solomon format is supported in push handlers at this time");

        const auto& project = handlerConfig.GetProject();
        const auto& service = handlerConfig.GetService();
        Y_ENSURE(!project.empty() && !service.empty(), "Project and service must be specified for handler");

        Server_->AddHandler(handlerConfig.GetEndpoint(), MakeHolder<TSolomonPushHandler>(project, service, writer));
    }
}

void THttpPushModule::Stop() {
    Server_->Stop();
}

void THttpPushModule::Start() {
    Server_->Start();
}

TAgentPushModulePtr CreateHttpPushModule(
    const THttpPushModuleConfig& config,
    IThreadPoolProvider& threadPoolProvider,
    IShardConsumerProvider* writer,
    IServerStatusListener* listener)
{
    return ::MakeHolder<THttpPushModule>(config, threadPoolProvider, writer, listener);
}

} // namespace NAgent
} // namespace NSolomon

