#include "http.h"

#include <solomon/agent/modules/pull/http/protos/http_pull_config.pb.h>
#include <solomon/agent/lib/http/headers.h>
#include <solomon/agent/misc/logger.h>

#include <library/cpp/http/client/client.h>
#include <library/cpp/http/fetch/exthttpcodes.h>
#include <library/cpp/http/misc/httpcodes.h>

#include <library/cpp/monlib/encode/format.h>
#include <library/cpp/monlib/encode/json/json.h>
#include <library/cpp/monlib/encode/prometheus/prometheus.h>
#include <library/cpp/monlib/encode/spack/spack_v1.h>

#include <util/datetime/base.h>
#include <util/stream/mem.h>
#include <util/string/builder.h>
#include <library/cpp/string_utils/url/url.h>


using namespace NMonitoring;

namespace NSolomon {
namespace NAgent {
namespace {

TMaybe<EFormat> ConvertFormat(THttpPullConfig::EFormat f) {
    switch (f) {
        case THttpPullConfig::JSON: return EFormat::JSON;
        case THttpPullConfig::SPACK: return EFormat::SPACK;
        case THttpPullConfig::PROMETHEUS: return EFormat::PROMETHEUS;
        default:
            return Nothing();
    }
}

class THttpPullModule: public IPullModule {
public:
    THttpPullModule(const TLabels& labels, const THttpPullConfig& conf)
        : Name_{"HttpPull/" + conf.GetUrl()}
        , Labels_{labels}
        , Format_{ConvertFormat(conf.GetFormat())}
    {
        const auto& url = conf.GetUrl();
        TStringBuf scheme, host;
        ui16 port;

        Y_ENSURE(!url.Empty(), "HttpPull module: URL cannot be empty");
        Y_ENSURE(TryGetSchemeHostAndPort(url, scheme, host, port), "HttpPull module: URL " << url << " is invalid");

        NHttp::TFetchOptions options;
        options.UserAgent = USER_AGENT_HEADER;
        options.RetryCount = conf.GetRetryCount();
        options.RetryDelay = TDuration::MilliSeconds(conf.GetRetryIntervalMillis());
        options.Timeout = TDuration::MilliSeconds(conf.GetTimeoutMillis() == 0
            ? 5000
            : conf.GetTimeoutMillis());

        TStringBuilder accept;
        accept << "Accept: ";

        auto fallback = [&]() {
            // let's tell about everything we support and work with whatever is sent back
            accept << NFormatContenType::SPACK
                    << ", " << NFormatContenType::JSON
                    << ", " << "text/plain"; // PROMETHEUS
        };

        if (!Format_) {
            fallback();
        } else {
            switch (*Format_) {
                case EFormat::SPACK:
                    accept << NFormatContenType::SPACK;
                    break;
                case EFormat::JSON:
                    accept << NFormatContenType::JSON;
                    break;
                case EFormat::PROMETHEUS:
                    accept << "text/plain";
                    break;
                case EFormat::PROTOBUF:
                case EFormat::TEXT:
                case EFormat::UNKNOWN:
                    fallback();
                    break;
            }
        }

        TVector<TString> headers;

        for (const auto& header: conf.GetHeaders()) {
            if (header.StartsWith("Accept:")) {
                SA_LOG(WARN) << "Found Accept header in config:\'" << header << "\', ignored";
                continue;
            }
            headers.push_back(header);
        }
        headers.push_back(accept);

        options.Method = THttpPullConfig::EHttpMethod_Name(conf.GetMethod());

        Query_ = NHttp::TFetchQuery{url, headers, options};
    }

private:
    TStringBuf Name() const override {
        return Name_;
    }

    int Pull(TInstant, IMetricConsumer* consumer) override {
        auto response = Fetch(Query_);
        if (auto code = response->Code; code != HTTP_OK) {
            TStringBuf errorReason = (code < 600) ? HttpCodeStrEx(code) : ExtHttpCodeStr(code);
            SA_LOG(ERROR) << "Request to " << Query_.GetUrl() << " failed with code " << code << ": " << errorReason;
            return 0;
        }

        EFormat format = Format_.Empty()
            ? FormatFromContentType(response->MimeType)
            : *Format_;

        if (format == EFormat::SPACK) {
            TMemoryInput in{response->Data};
            ConsumeCommonLabels(consumer);
            DecodeSpackV1(&in, consumer);
        } else if (format == EFormat::JSON) {
            ConsumeCommonLabels(consumer);
            DecodeJson(response->Data, consumer);
        } else if (format == EFormat::PROMETHEUS) {
            ConsumeCommonLabels(consumer);
            DecodePrometheus(response->Data, consumer);
        } else {
            ythrow yexception() << "Response from " << Query_.GetUrl()
                                << " has unsupported format: " << format
                                << ", Content-Type: " << response->MimeType;
        }

        return 0;
    }

    void ConsumeCommonLabels(IMetricConsumer* consumer) {
        consumer->OnLabelsBegin();
        for (auto&& label: Labels_) {
            consumer->OnLabel(TString{label.Name()}, TString{label.Value()});
        }
        consumer->OnLabelsEnd();
    }

private:
    NHttp::TFetchQuery Query_;
    const TString Name_;
    const TLabels Labels_;
    TMaybe<EFormat> Format_;
};

} // namespace


IPullModulePtr CreateHttpPullModule(const TLabels& labels, const THttpPullConfig& config) {
    return new THttpPullModule(labels, config);
}

} // namespace NAgent
} // namespace NSolomon
