#pragma once

#include <library/cpp/http/misc/parsed_request.h>
#include <library/cpp/http/server/http.h>
#include <library/cpp/http/server/response.h>
#include <library/cpp/testing/common/network.h>
#include <library/cpp/testing/gtest/gtest.h>
#include <library/cpp/testing/mock_server/server.h>

#include <util/generic/yexception.h>
#include <util/system/yassert.h>

#include <functional>

namespace NSolomon::NAgent {
    struct TMockConfig {
        using THeadersHandler = std::function<bool(const THttpHeaders&)>;

        TString AllowedMethod = "GET";
        TString ResponseBody;
        THttpHeaders Headers;
        THeadersHandler HeadersHandler;
    };

    class TMockReplier: public TRequestReplier {
    public:
        TMockReplier(const TMockConfig& config)
            : Config_{config}
        {
        }

        bool DoReply(const TReplyParams& params) override {
            TParsedHttpFull req(params.Input.FirstLine());
            if (req.Method != Config_.AllowedMethod) {
                ythrow yexception() << "The requested method: \"" << req.Method << "\" is not allowed";
            }

            if (req.Method == "GET") {
                Y_ENSURE(params.Input.ReadAll().Empty());
            }

            Y_ENSURE(Config_.HeadersHandler(params.Input.Headers()));

            THttpResponse response(HTTP_OK);
            response.SetContent(Config_.ResponseBody);
            response.AddMultipleHeaders(Config_.Headers);
            response.OutTo(params.Output);

            return true;
        }

    private:
        const TMockConfig& Config_;
    };

    class TMockHttpServer {
    public:
        explicit TMockHttpServer(TMockConfig config)
            : Config_{std::move(config)}
            , Port_{NTesting::GetFreePort()}
        {
            Server_.Reset(new NMock::TMockServer(Port_, [this]() {
                return new TMockReplier(Config_);
            }));
        }

        ui16 GetPort() {
            return Port_;
        }

    private:
        TMockConfig Config_;
        NTesting::TPortHolder Port_;
        THolder<NMock::TMockServer> Server_;
    };
} // namespace NSolomon::NAgent
