#include "http_mock_server.h"

#include <solomon/agent/lib/module/pull_module.h>
#include <solomon/agent/misc/labels.h>
#include <solomon/agent/modules/pull/http/http.h>
#include <solomon/agent/modules/pull/http/protos/http_pull_config.pb.h>
#include <solomon/agent/protos/service_config.pb.h>

#include <library/cpp/http/io/headers.h>
#include <library/cpp/monlib/consumers/collecting_consumer.h>
#include <library/cpp/testing/common/network.h>
#include <library/cpp/testing/gtest/gtest.h>
#include <library/cpp/testing/mock_server/server.h>

#include <util/datetime/base.h>
#include <util/generic/fwd.h>
#include <util/generic/ptr.h>
#include <util/string/builder.h>

using namespace NMonitoring;
using namespace NSolomon;
using namespace NAgent;

THolder<NMock::TMockServer> CreateMockServer(
    ui16 port,
    TString allowedMethod,
    TMockConfig::THeadersHandler handler = [](const THttpHeaders&) { return true; })
{
    TMockConfig config;
    config.ResponseBody = "{}";
    config.Headers.AddHeader(THttpInputHeader("Content-Type: application/json"));
    config.AllowedMethod = std::move(allowedMethod);
    config.HeadersHandler = std::move(handler);

    return MakeHolder<NMock::TMockServer>(port, [config]() { return new TMockReplier(config); });
}

THttpPullConfig ConstructBaseConfig(ui16 port) {
    THttpPullConfig config;
    config.SetUrl(TStringBuilder() << "http://localhost:" << port << "/");
    config.SetRetryIntervalMillis(100);
    config.SetFormat(THttpPullConfig::JSON);
    return config;
}

void ValidateConsumer(const TCollectingConsumer& consumer) {
    ASSERT_TRUE(consumer.Metrics.empty());
    ASSERT_TRUE(consumer.CommonTime == TInstant::Zero());
    ASSERT_TRUE(consumer.CommonLabels.empty());
}

TEST(TPullAgentTest, EmptyGetTest) {
    auto port = NTesting::GetFreePort();
    THolder<NMock::TMockServer> server = CreateMockServer(port, "GET");

    THttpPullConfig config = ConstructBaseConfig(port);
    NAgent::TLabels labels;
    TCollectingConsumer consumer;

    IPullModulePtr pullModule = CreateHttpPullModule(labels, config);
    pullModule->Pull(TInstant::Zero(), &consumer);
    ValidateConsumer(consumer);
}

TEST(TPullAgentTest, HeaderGetTest) {
    auto port = NTesting::GetFreePort();
    THolder<NMock::TMockServer> server = CreateMockServer(port, "GET", [](const THttpHeaders& headers) {
        const THttpInputHeader* test_header = headers.FindHeader("Test");
        if (test_header == nullptr) {
            return false;
        }

        return test_header->Value() == "test";
    });

    THttpPullConfig config = ConstructBaseConfig(port);
    config.AddHeaders("Test: test");

    NAgent::TLabels labels;
    TCollectingConsumer consumer;

    IPullModulePtr pullModule = CreateHttpPullModule(labels, config);
    pullModule->Pull(TInstant::Zero(), &consumer);
    ValidateConsumer(consumer);
}

TEST(TPullAgentTest, EmptyPostTest) {
    auto port = NTesting::GetFreePort();
    THolder<NMock::TMockServer> server = CreateMockServer(port, "GET");

    THttpPullConfig config = ConstructBaseConfig(port);
    config.SetMethod(THttpPullConfig::POST);

    NAgent::TLabels labels;
    TCollectingConsumer consumer;

    IPullModulePtr pullModule = CreateHttpPullModule(labels, config);
    pullModule->Pull(TInstant::Zero(), &consumer);
    ValidateConsumer(consumer);
}

TEST(TPullAgentTest, HeaderPostTest) {
    auto port = NTesting::GetFreePort();
    THolder<NMock::TMockServer> server = CreateMockServer(port, "POST", [](const THttpHeaders& headers) {
        const THttpInputHeader* test_header = headers.FindHeader("Test");
        if (test_header == nullptr) {
            return false;
        }

        return test_header->Value() == "test";
    });

    THttpPullConfig config = ConstructBaseConfig(port);
    config.AddHeaders("Test: test");
    config.SetMethod(THttpPullConfig::POST);

    NAgent::TLabels labels;
    TCollectingConsumer consumer;

    IPullModulePtr pullModule = CreateHttpPullModule(labels, config);
    pullModule->Pull(TInstant::Zero(), &consumer);
    ValidateConsumer(consumer);
}
