#include "nvidia_gpu.h"

#include <solomon/agent/misc/logger.h>
#include <solomon/agent/modules/pull/nvidia_gpu/protos/nvidia_gpu_pull_config.pb.h>

#include <library/cpp/monlib/metrics/metric_type.h>

#include <util/datetime/base.h>
#include <util/generic/maybe.h>
#include <util/generic/string.h>
#include <util/stream/mem.h>
#include <util/stream/pipe.h>
#include <util/string/split.h>
#include <util/system/fs.h>

namespace NSolomon::NAgent {

namespace {
#ifdef _unix_
    constexpr TStringBuf DEFAULT_NVIDIA_SMI_PATH = "/usr/bin";
#else
    constexpr TStringBuf DEFAULT_NVIDIA_SMI_PATH = "%ProgramW6432%\\NVIDIA Corporation\\NVSMI";
#endif

#ifdef _unix_
    constexpr TStringBuf DEFAULT_NVIDIA_SMI_BINARY = "nvidia-smi";
#else
    constexpr TStringBuf DEFAULT_NVIDIA_SMI_BINARY = "nvidia-smi.exe";
#endif

#ifdef _unix_
    constexpr TStringBuf FOLDER_SEPARATOR = "/";
#else
    constexpr TStringBuf FOLDER_SEPARATOR = "\\";
#endif

    struct TNvidiaGpuSettings {
        TString GetPowerAndUsageDataOnce = "dmon -c 1 -s pu -i";    // see nvidia-smi -h for refs
        TStringBuf DefaultSmiPath = DEFAULT_NVIDIA_SMI_PATH;
        TStringBuf DefaultBinaryName = DEFAULT_NVIDIA_SMI_BINARY;
        TStringBuf DefaultSeparator = FOLDER_SEPARATOR;
    };

    class TNvidiaSmiConnector {
    public:
        explicit TNvidiaSmiConnector(const TNvidiaGpuConfig& config, const TNvidiaGpuSettings& settings)
            : Config_(config)
            , NvidiaGpuSettings_(settings)
        {
            TStringBuf path = !Config_.GetPath().empty() ? Config_.GetPath() : settings.DefaultSmiPath;
            auto binary = TString::Join(path, settings.DefaultSeparator, settings.DefaultBinaryName);
            if (!NFs::Exists(binary)) {
                ythrow yexception() << "Nvidia-smi not found by path: " << binary;
            }

            Command_ = TString::Join(binary, " ", settings.GetPowerAndUsageDataOnce, " ", ToString(config.GetGpuId()));
        }

        void GetMetrics(NMonitoring::IMetricConsumer* consumer) {
            GetMetricsImpl(consumer, Config_);
        }

    private:
        TString CallSmi(const TString& command) {
            TString result;

            try {
                TPipeInput pipe(command);
                result = pipe.ReadAll();
            } catch (const yexception& e) {
                SA_LOG(WARN) << "Failed to call Nvidia-smi. Error: " << e.what();
            }

            return result;
        }

        void FillMetrics(TMemoryInput& metrics, TMemoryInput& values) {
            while (true) {
                TString name, value;
                metrics >> name;
                if (name.empty())
                    break;

                if (name == "#")
                    continue;

                values >> value;
                Metrics_[std::move(name)] = FromString<double>(value);
            }
        }

        bool IsExpectedFormat(TStringBuf line) {
            return !line.empty() && line[0] == '#';
        }

        bool ParseSmiAnswer(const TString& answer) {
            TMemoryInput metrics, values;
            int count = 0;
            for (TStringBuf line: StringSplitter(answer).Split('\n').SkipEmpty()) {
                if (count == 0 && !IsExpectedFormat(line)) {
                    SA_LOG(WARN) << "Unexpected format of answer of Nvidia-smi: " << answer;
                    Metrics_.clear();
                    return false;
                } else if (count == 0) {
                    metrics = TMemoryInput(line);
                } else if (count == 2) {
                    values = TMemoryInput(line);
                } else if (count > 2) {
                    break;
                }
                ++count;
            }
            FillMetrics(metrics, values);
            return true;
        }

        void OutputGauge(NMonitoring::IMetricConsumer* consumer, TStringBuf metricName, TStringBuf metricInnerName) {
            if (auto it = Metrics_.find(metricInnerName); it != Metrics_.end()) {
                consumer->OnMetricBegin(NMonitoring::EMetricType::GAUGE);
                {
                    consumer->OnLabelsBegin();
                    consumer->OnLabel("sensor", metricName);
                    consumer->OnLabelsEnd();
                }
                consumer->OnDouble(TInstant::Zero(), it->second);
                consumer->OnMetricEnd();
            } else {
                SA_LOG(WARN) << "Not found or unexpected InnerName: '" << metricInnerName << "' for metric: '" << metricName << "'.";
            }
        }

        inline void GetMetricsImpl(NMonitoring::IMetricConsumer* consumer, const TNvidiaGpuConfig& config) {
            if (!ParseSmiAnswer(CallSmi(Command_))) {
                SA_LOG(WARN) << "Failed to make query to Nvidia-smi.";
                return;
            }

            if (config.GetGpu() == TNvidiaGpuConfig::ON) {
                OutputGauge(consumer, "gpuUsagePercents", "sm");
                OutputGauge(consumer, "gpuTemperatureCelsius", "gtemp");
            }

            if (config.GetMemory() == TNvidiaGpuConfig::ON) {
                OutputGauge(consumer, "memoryUsagePercents", "mem");
                OutputGauge(consumer, "memoryTemperatureCelsius", "mtemp");
            }
        }

    private:
        TString Command_;
        const TNvidiaGpuConfig& Config_;
        TNvidiaGpuSettings NvidiaGpuSettings_;
        THashMap<TString, double> Metrics_;
    };


    class TNvidiaGpuPullModule : public IPullModule {
    public:
        TNvidiaGpuPullModule(const TLabels& labels, const TNvidiaGpuConfig& config)
            : Labels_{ std::move(labels) }
            , Config_(config)
        {
        }

        TStringBuf Name() const override {
            return TStringBuf("NvidiaGpu");
        }

        // non-zero return code indicates that this module should not be called anymore
        int Pull(TInstant, NMonitoring::IMetricConsumer* consumer) override {
            consumer->OnLabelsBegin();
            for (auto&& label: Labels_) {
                consumer->OnLabel(label.Name(), label.Value());
            }
            consumer->OnLabelsEnd();

            TNvidiaSmiConnector connector(Config_, TNvidiaGpuSettings{});
            connector.GetMetrics(consumer);

            return 0;
        }

    private:
        const TLabels Labels_;
        const TNvidiaGpuConfig Config_;
    };
}  // namespace

    IPullModulePtr CreateNvidiaGpuPullModule(const TLabels& labels, const TNvidiaGpuConfig& config) {
        return new TNvidiaGpuPullModule(labels, config);
    }
}  // namespace NSolomon::NAgent
