#include "py_cast.h"

#include <solomon/agent/lib/python2/error.h>
#include <solomon/agent/lib/python2/py_string.h>

#include <util/generic/ymath.h>

#include <type_traits>

namespace NSolomon {
namespace NAgent {

namespace {

template <typename TConsumer>
void PyDictForEachTmpl(const PyObject* o, TConsumer c) {
    PY_ENSURE_TYPE(PyDict_Check, o, "expected to get dict");

    Py_ssize_t pos = 0;
    PyObject *namePy, *valuePy;

    auto nco = const_cast<PyObject*>(o); // non const object
    while (PyDict_Next(nco, &pos, &namePy, &valuePy)) {
        PY_ENSURE_TYPE(PyString_Check, namePy, "label name must be a string");
        TStringBuf name = NPython2::ToStringBufNoAlloc(namePy);

        PY_ENSURE_TYPE(PyString_Check, valuePy, "label value must be a string");
        TStringBuf value = NPython2::ToStringBufNoAlloc(valuePy);

        c(name, value);
    }
}

} // namespace

void PyDictToLabels(const PyObject* o, NMonitoring::TLabels* labels) {
    PyDictForEachTmpl(o, [labels](TStringBuf name, TStringBuf value) {
        labels->Add(name, value);
    });
}

void PyDictForEach(const PyObject* o, std::function<void(TStringBuf, TStringBuf)> fn) {
    PyDictForEachTmpl(o, std::move(fn));
}

template <typename T, typename F>
TVector<T> PyListToVector(const PyObject* o, F&& converter) {
    static_assert(std::is_same<
        typename std::invoke_result<F, PyObject*>::type,
        typename std::decay<T>::type
    >::value);

    auto* pyList = const_cast<PyObject*>(o);
    PY_ENSURE_TYPE(PyList_Check, pyList, "must be a list");

    const auto len = PyList_Size(pyList);

    TVector<T> result;
    result.reserve(len);
    for (Py_ssize_t i = 0; i < len; ++i) {
        const auto pyVal = PyList_GetItem(pyList, i);
        result.push_back(converter(pyVal));
    }

    return result;
}

TVector<double> PyListToVectorFloat(const PyObject* o) {
    return PyListToVector<double>(o, PyNumberToDouble);
}

TVector<ui64> PyListToVectorUi64(const PyObject* o) {
    return PyListToVector<ui64>(o, PyNumberToUnsignedLong);
}

double PyNumberToDouble(const PyObject* o) {
    auto nco = const_cast<PyObject*>(o); // non const object
    if (PyFloat_Check(nco)) {
        return PyFloat_AsDouble(nco);
    } else if (PyInt_Check(nco)) {
        return static_cast<double>(PyInt_AsLong(nco));
    } else if (PyLong_Check(nco)) {
        return PyLong_AsDouble(nco);
    }

    PY_FAIL("value must be Float, Int or Long, but got: " << NPython2::TypeName(o));
}

PyObject* DoubleToPyNumber(double value) {
    return PyFloat_FromDouble(value);
}

template <typename T, typename TFromLong, typename TFromInt>
T PyNumberToInteger(const PyObject* o, TFromLong longConv, TFromInt intConv) {
    auto nco = const_cast<PyObject*>(o); // non const object
    if (PyLong_Check(nco)) {
        return longConv(nco);
    } else if (PyInt_Check(nco)) {
        return static_cast<T>(intConv(nco));
    }

    PY_FAIL("value must be Long or Int, but got: " << NPython2::TypeName(o));
}

i64 PyNumberToLong(const PyObject* o) {
    return PyNumberToInteger<i64>(o, PyLong_AsLongLong, PyInt_AsLong);
}

PyObject* LongToPyNumber(i64 value) {
    return PyLong_FromLongLong(value);
}

ui32 PyNumberToUnsignedInt32(const PyObject* o) {
    const auto val = PyNumberToInteger<ui32>(o, PyLong_AsUnsignedLongLong, PyInt_AsUnsignedLongLongMask);
    Y_ENSURE(val <= std::numeric_limits<ui32>::max(), "value " << val << " is out of ui32 bounds");
    return val;
}

ui64 PyNumberToUnsignedLong(const PyObject* o) {
    return PyNumberToInteger<ui64>(o, PyLong_AsUnsignedLongLong, PyInt_AsUnsignedLongLongMask);
}

PyObject* UnsignedLongToPyNumber(ui64 value) {
    return PyLong_FromUnsignedLongLong(value);
}

TInstant PyNumberToInstant(const PyObject* o) {
    if (o == nullptr) {
        return TInstant::Zero();
    }

    auto nco = const_cast<PyObject*>(o); // non const object
    if (PyInt_Check(nco)) {
        return TInstant::MilliSeconds(PyInt_AsUnsignedLongLongMask(nco));
    } else if (PyLong_Check(nco)) {
        return TInstant::MilliSeconds(PyLong_AsUnsignedLongLong(nco));
    } else if (PyFloat_Check(o)) {
        // python's time.time() returns current time as Float with
        // fractions of second
        double ts = PyFloat_AsDouble(nco);
        return TInstant::MilliSeconds(static_cast<ui64>(Abs(ts) * 1000));
    }

    PY_FAIL("timestamp must be Long or Float, but got: " << NPython2::TypeName(o));
}

PyObject* InstantToPyNumber(TInstant value) {
    return PyLong_FromUnsignedLongLong(value.MilliSeconds());
}

} // namespace NAgent
} // namespace NSolomon
