#include "py_histogram.h"
#include "py_cast.h"

#include <solomon/agent/lib/python2/ptr.h>
#include <solomon/agent/lib/python2/error.h>

#include <library/cpp/monlib/metrics/metric_registry.h>

namespace NSolomon::NAgent {

using namespace NMonitoring;

IHistogramCollectorPtr CreateHistCollectorFromPyArgs(PyObject* args, PyObject* kwargs, NMonitoring::TLabels* labels) {
    static NPython2::TObjectPtr EMPTY_TUPLE = PyTuple_New(0);

    PyObject* labelsPy {nullptr};
    char* type {nullptr};

    PyObject* buckets {nullptr};
    int bucketCount {0};
    double base {0.};
    double scale {1.};
    i64 startValue{0};
    i64 bucketWidth{0};

    static const char* kwlist[] = {
        "type", "buckets", "bucket_count",
        "base", "scale",
        "start_value", "bucket_width",
        nullptr};

    static const char* labelsKwlist[] = {
        "labels", "type", "buckets", "bucket_count",
        "base", "scale",
        "start_value", "bucket_width",
        nullptr};

    bool ok{};
    if (labels == nullptr) {
        char** kws = const_cast<char**>(kwlist);

        ok = PyArg_ParseTupleAndKeywords(EMPTY_TUPLE.Get(), kwargs, "s|Oiddll", kws,
            &type, &buckets, &bucketCount, &base, &scale, &startValue, &bucketWidth);
    } else {
        char** kws = const_cast<char**>(labelsKwlist);

        ok = PyArg_ParseTupleAndKeywords(args, kwargs, "Os|Oiddll", kws,
        &labelsPy, &type, &buckets, &bucketCount, &base, &scale, &startValue, &bucketWidth);
    }

    if (!ok) {
        return nullptr;
    }

    constexpr TStringBuf EXPLICIT = "explicit";
    constexpr TStringBuf EXPONENTIAL = "exponential";
    constexpr TStringBuf LINEAR = "linear";
    constexpr auto MIN_START_VAL = std::numeric_limits<decltype(startValue)>::min();

    NMonitoring::IHistogramCollectorPtr collector;
    if (EXPLICIT == type) {
        auto bucketBounds = PyListToVectorFloat(buckets);
        PY_ENSURE(bucketBounds.size() > 0, "At least one bucket must be specified for an explicit histogram");
        collector = NMonitoring::ExplicitHistogram(std::move(bucketBounds));
    } else if (EXPONENTIAL == type) {
        PY_ENSURE(bucketCount >= 2, "bucket_count must be specified for an exponential histogram and be greater than 1");
        PY_ENSURE(base, "base must be specified for an exponential histogram and be greater than 0");
        // scale has a default value
        collector = NMonitoring::ExponentialHistogram(bucketCount, base, scale);
    } else if (LINEAR == type) {
        PY_ENSURE(bucketCount >= 2, "bucket_count must be specified for an linear histogram and be greater than 1");
        PY_ENSURE(startValue > MIN_START_VAL, "bucket_count must be specified for an linear histogram");
        PY_ENSURE(bucketWidth >= 1, "bucket_width must be specified for an exponential histogram and be greater than or equal to 1");
        collector = NMonitoring::LinearHistogram(bucketCount, startValue, bucketWidth);
    } else {
        ythrow yexception() << "Unknown histogram type " << type << ", " <<
            "allowed values are explicit, exponential and linear";
    }

    if (labels != nullptr) {
        PyDictToLabels(labelsPy, labels);
    }

    return collector;
}

} // namespace NSolomon::NAgent
