#include "py_cast.h"
#include "py_metric.h"

#include <solomon/agent/lib/python2/error.h>

#include <library/cpp/monlib/metrics/metric.h>

#include <util/string/builder.h>

#include <Python.h>

namespace NSolomon {
namespace NAgent {
namespace {


template <typename TDerived, typename TNativeMetric>
struct TPyMetric {
    PyObject_HEAD
    TNativeMetric* Delegate;

    static TDerived* Cast(PyObject* o) {
        return reinterpret_cast<TDerived*>(o);
    }

    static void Dealloc(PyObject* self) {
        delete Cast(self);
    }

    static PyObject* New(TNativeMetric* delegate) {
        auto* metric = new TDerived;
        PyObject_INIT(metric, TDerived::Type());
        metric->Delegate = delegate;
        return reinterpret_cast<PyObject*>(metric);
    }
};

///////////////////////////////////////////////////////////////////////////////
// TPyGauge
///////////////////////////////////////////////////////////////////////////////
struct TPyGauge: TPyMetric<TPyGauge, NMonitoring::TGauge> {
    static PyTypeObject* Type() {
        return &PyGaugeType;
    }

    static PyObject* Repr(PyObject* self) {
        return PyString_FromFormat("<solomon.Gauge object at %p>", self);
    }

    static PyObject* Set(PyObject* self, PyObject* valuePy) {
        PY_TRY {
            double value = PyNumberToDouble(valuePy);
            Cast(self)->Delegate->Set(value);
            Py_RETURN_NONE;
        } PY_CATCH(nullptr)
    }

    static PyObject* Get(PyObject* self, PyObject* /*args*/) {
        PY_TRY {
            double value = Cast(self)->Delegate->Get();
            return DoubleToPyNumber(value);
        } PY_CATCH(nullptr)
    }
};

PyDoc_STRVAR(PyGauge_Doc,
    "TODO: add solomon.Gauge docs");
PyDoc_STRVAR(PyGaugeSet_Doc,
    "TODO: add solomon.Gauge::set(value) docs");
PyDoc_STRVAR(PyGaugeGet_Doc,
    "TODO: add solomon.Gauge::get() docs");

static PyMethodDef PyGaugeMethods[] = {
    { "set",  TPyGauge::Set, METH_O, PyGaugeSet_Doc },
    { "get",  TPyGauge::Get, METH_NOARGS, PyGaugeGet_Doc },
    { nullptr, nullptr, 0, nullptr }    /* sentinel */
};


///////////////////////////////////////////////////////////////////////////////
// TPyGauge
///////////////////////////////////////////////////////////////////////////////
struct TPyIntGauge: TPyMetric<TPyIntGauge, NMonitoring::TIntGauge> {
    static PyTypeObject* Type() {
        return &PyIntGaugeType;
    }

    static PyObject* Repr(PyObject* self) {
        return PyString_FromFormat("<solomon.IntGauge object at %p>", self);
    }

    static PyObject* Set(PyObject* self, PyObject* valuePy) {
        PY_TRY {
            i64 value = PyNumberToLong(valuePy);
            Cast(self)->Delegate->Set(value);
            Py_RETURN_NONE;
        } PY_CATCH(nullptr)
    }

    static PyObject* Get(PyObject* self, PyObject* /*args*/) {
        PY_TRY {
            double value = Cast(self)->Delegate->Get();
            return DoubleToPyNumber(value);
        } PY_CATCH(nullptr)
    }
};

PyDoc_STRVAR(PyIntGauge_Doc,
    "Creates a new instant of integer gauge metric");
PyDoc_STRVAR(PyIntGaugeSet_Doc,
    "Sets metric value");
PyDoc_STRVAR(PyIntGaugeGet_Doc,
    "Gets metric value");

static PyMethodDef PyIntGaugeMethods[] = {
    { "set",  TPyIntGauge::Set, METH_O, PyIntGaugeSet_Doc },
    { "get",  TPyIntGauge::Get, METH_NOARGS, PyIntGaugeGet_Doc },
    { nullptr, nullptr, 0, nullptr }    /* sentinel */
};

///////////////////////////////////////////////////////////////////////////////
// TPyCounter
///////////////////////////////////////////////////////////////////////////////
struct TPyCounter: TPyMetric<TPyCounter, NMonitoring::TCounter> {
    static PyTypeObject* Type() {
        return &PyCounterType;
    }

    static PyObject* Repr(PyObject* self) {
        return PyString_FromFormat("<solomon.Counter object at %p>", self);
    }

    static PyObject* Inc(PyObject* self, PyObject* /*args*/) {
        PY_TRY {
            ui64 value = Cast(self)->Delegate->Inc();
            return UnsignedLongToPyNumber(value);
        } PY_CATCH(nullptr)
    }

    static PyObject* Add(PyObject* self, PyObject* valuePy) {
        PY_TRY {
            ui64 value = PyNumberToUnsignedLong(valuePy);
            value = Cast(self)->Delegate->Add(value);
            return UnsignedLongToPyNumber(value);
        } PY_CATCH(nullptr)
    }

    static PyObject* Get(PyObject* self, PyObject* /*args*/) {
        PY_TRY {
            ui64 value = Cast(self)->Delegate->Get();
            return UnsignedLongToPyNumber(value);
        } PY_CATCH(nullptr)
    }
};

PyDoc_STRVAR(PyCounter_Doc,
    "TODO: add solomon.Counter docs");
PyDoc_STRVAR(PyCounterInc_Doc,
    "TODO: add solomon.Counter::inc() docs");
PyDoc_STRVAR(PyCounterAdd_Doc,
    "TODO: add solomon.Counter::add(value) docs");
PyDoc_STRVAR(PyCounterGet_Doc,
    "TODO: add solomon.Counter::get() docs");

static PyMethodDef PyCounterMethods[] = {
    { "inc",  TPyCounter::Inc, METH_NOARGS, PyCounterInc_Doc },
    { "add",  TPyCounter::Add, METH_O, PyCounterAdd_Doc },
    { "get",  TPyCounter::Get, METH_NOARGS, PyCounterGet_Doc },
    { nullptr, nullptr, 0, nullptr }    /* sentinel */
};

///////////////////////////////////////////////////////////////////////////////
// TPyRate
///////////////////////////////////////////////////////////////////////////////
struct TPyRate: TPyMetric<TPyRate, NMonitoring::TRate> {
    static PyTypeObject* Type() {
        return &PyRateType;
    }

    static PyObject* Repr(PyObject* self) {
        return PyString_FromFormat("<solomon.Rate object at %p>", self);
    }

    static PyObject* Inc(PyObject* self, PyObject* /*args*/) {
        PY_TRY {
            ui64 value = Cast(self)->Delegate->Inc();
            return UnsignedLongToPyNumber(value);
        } PY_CATCH(nullptr)
    }

    static PyObject* Add(PyObject* self, PyObject* valuePy) {
        PY_TRY {
            ui64 value = PyNumberToUnsignedLong(valuePy);
            value = Cast(self)->Delegate->Add(value);
            return UnsignedLongToPyNumber(value);
        } PY_CATCH(nullptr)
    }

    static PyObject* Get(PyObject* self, PyObject* /*args*/) {
        PY_TRY {
            ui64 value = Cast(self)->Delegate->Get();
            return UnsignedLongToPyNumber(value);
        } PY_CATCH(nullptr)
    }
};

PyDoc_STRVAR(PyRate_Doc,
    "TODO: add solomon.Rate docs");
PyDoc_STRVAR(PyRateInc_Doc,
    "TODO: add solomon.Rate::inc() docs");
PyDoc_STRVAR(PyRateAdd_Doc,
    "TODO: add solomon.Rate::add(value) docs");
PyDoc_STRVAR(PyRateGet_Doc,
    "TODO: add solomon.Rate::get() docs");

static PyMethodDef PyRateMethods[] = {
    { "inc",  TPyRate::Inc, METH_NOARGS, PyRateInc_Doc },
    { "add",  TPyRate::Add, METH_O, PyRateAdd_Doc },
    { "get",  TPyRate::Get, METH_NOARGS, PyRateGet_Doc },
    { nullptr, nullptr, 0, nullptr }    /* sentinel */
};


///////////////////////////////////////////////////////////////////////////////
// TPyHistogram
///////////////////////////////////////////////////////////////////////////////
struct TPyHistogram: TPyMetric<TPyHistogram, NMonitoring::THistogram> {
    static PyTypeObject* Type() {
        return &PyHistogramType;
    }

    static PyObject* Repr(PyObject* self) {
        return PyString_FromFormat("<solomon.Histogram object at %p>", self);
    }

    static PyObject* Collect(PyObject* self, PyObject* args) {
        PY_TRY {
            i64 value {0};
            ui32 count {1};

            PY_ENSURE_TYPE(PyTuple_Check, args, "args are not tuple");


            const auto len = PyTuple_GET_SIZE(args);
            if (len < 1 || len > 2) {
                const TString err  = TStringBuilder() << "expected 1 or 2 arguments, but found " << len;
                PyErr_SetString(PyExc_ValueError, err.c_str());
                return nullptr;
            }

            auto* pyValue = PyTuple_GET_ITEM(args, 0);
            value = PyNumberToLong(pyValue);

            if (len > 1) {
                auto* pyValue = PyTuple_GET_ITEM(args, 1);
                count = PyNumberToUnsignedInt32(pyValue);
            }

            Y_ENSURE(count > 0, "count must be greater than 0");
            Cast(self)->Delegate->Record(value, count);
            Py_RETURN_NONE;
        } PY_CATCH(nullptr)
    }
};

PyDoc_STRVAR(PyHistogram_Doc,
    "");
PyDoc_STRVAR(PyHistogramCollect_Doc,
    "Add a value to the histogram");

static PyMethodDef PyHistogramMethods[] = {
    { "collect",  TPyHistogram::Collect, METH_VARARGS, PyHistogramCollect_Doc },
    { nullptr, nullptr, 0, nullptr }    /* sentinel */
};


} // namespace


PyTypeObject PyIntGaugeType = {
    PyVarObject_HEAD_INIT(&PyType_Type, 0)
    /* tp_name           */ "solomon.Gauge",
    /* tp_basicsize      */ sizeof(TPyIntGauge),
    /* tp_itemsize       */ 0,
    /* tp_dealloc        */ TPyIntGauge::Dealloc,
    /* tp_print          */ nullptr,
    /* tp_getattr        */ nullptr,
    /* tp_setattr        */ nullptr,
    /* tp_compare        */ nullptr,
    /* tp_repr           */ TPyIntGauge::Repr,
    /* tp_as_number      */ nullptr,
    /* tp_as_sequence    */ nullptr,
    /* tp_as_mapping     */ nullptr,
    /* tp_hash           */ nullptr,
    /* tp_call           */ nullptr,
    /* tp_str            */ nullptr,
    /* tp_getattro       */ PyObject_GenericGetAttr,
    /* tp_setattro       */ nullptr,
    /* tp_as_buffer      */ nullptr,
    /* tp_flags          */ 0,
    /* tp_doc            */ PyIntGauge_Doc,
    /* tp_traverse       */ nullptr,
    /* tp_clear          */ nullptr,
    /* tp_richcompare    */ nullptr,
    /* tp_weaklistoffset */ 0,
    /* tp_iter           */ nullptr,
    /* tp_iternext       */ nullptr,
    /* tp_methods        */ PyIntGaugeMethods,
    /* tp_members        */ nullptr,
    /* tp_getset         */ nullptr,
    /* tp_base           */ nullptr,
    /* tp_dict           */ nullptr,
    /* tp_descr_get      */ nullptr,
    /* tp_descr_set      */ nullptr,
    /* tp_dictoffset     */ 0,
    /* tp_init           */ nullptr,
    /* tp_alloc          */ nullptr,
    /* tp_new            */ nullptr,
    /* tp_free           */ nullptr,
    /* tp_is_gc          */ nullptr,
    /* tp_bases          */ nullptr,
    /* tp_mro            */ nullptr,
    /* tp_cache          */ nullptr,
    /* tp_subclasses     */ nullptr,
    /* tp_weaklist       */ nullptr,
    /* tp_del            */ nullptr,
    /* tp_version_tag    */ 0,
};

PyTypeObject PyGaugeType = {
    PyVarObject_HEAD_INIT(&PyType_Type, 0)
    /* tp_name           */ "solomon.Gauge",
    /* tp_basicsize      */ sizeof(TPyGauge),
    /* tp_itemsize       */ 0,
    /* tp_dealloc        */ TPyGauge::Dealloc,
    /* tp_print          */ nullptr,
    /* tp_getattr        */ nullptr,
    /* tp_setattr        */ nullptr,
    /* tp_compare        */ nullptr,
    /* tp_repr           */ TPyGauge::Repr,
    /* tp_as_number      */ nullptr,
    /* tp_as_sequence    */ nullptr,
    /* tp_as_mapping     */ nullptr,
    /* tp_hash           */ nullptr,
    /* tp_call           */ nullptr,
    /* tp_str            */ nullptr,
    /* tp_getattro       */ PyObject_GenericGetAttr,
    /* tp_setattro       */ nullptr,
    /* tp_as_buffer      */ nullptr,
    /* tp_flags          */ 0,
    /* tp_doc            */ PyGauge_Doc,
    /* tp_traverse       */ nullptr,
    /* tp_clear          */ nullptr,
    /* tp_richcompare    */ nullptr,
    /* tp_weaklistoffset */ 0,
    /* tp_iter           */ nullptr,
    /* tp_iternext       */ nullptr,
    /* tp_methods        */ PyGaugeMethods,
    /* tp_members        */ nullptr,
    /* tp_getset         */ nullptr,
    /* tp_base           */ nullptr,
    /* tp_dict           */ nullptr,
    /* tp_descr_get      */ nullptr,
    /* tp_descr_set      */ nullptr,
    /* tp_dictoffset     */ 0,
    /* tp_init           */ nullptr,
    /* tp_alloc          */ nullptr,
    /* tp_new            */ nullptr,
    /* tp_free           */ nullptr,
    /* tp_is_gc          */ nullptr,
    /* tp_bases          */ nullptr,
    /* tp_mro            */ nullptr,
    /* tp_cache          */ nullptr,
    /* tp_subclasses     */ nullptr,
    /* tp_weaklist       */ nullptr,
    /* tp_del            */ nullptr,
    /* tp_version_tag    */ 0,
};

PyTypeObject PyCounterType = {
    PyVarObject_HEAD_INIT(&PyType_Type, 0)
    /* tp_name           */ "solomon.Counter",
    /* tp_basicsize      */ sizeof(TPyCounter),
    /* tp_itemsize       */ 0,
    /* tp_dealloc        */ TPyCounter::Dealloc,
    /* tp_print          */ nullptr,
    /* tp_getattr        */ nullptr,
    /* tp_setattr        */ nullptr,
    /* tp_compare        */ nullptr,
    /* tp_repr           */ TPyCounter::Repr,
    /* tp_as_number      */ nullptr,
    /* tp_as_sequence    */ nullptr,
    /* tp_as_mapping     */ nullptr,
    /* tp_hash           */ nullptr,
    /* tp_call           */ nullptr,
    /* tp_str            */ nullptr,
    /* tp_getattro       */ PyObject_GenericGetAttr,
    /* tp_setattro       */ nullptr,
    /* tp_as_buffer      */ nullptr,
    /* tp_flags          */ 0,
    /* tp_doc            */ PyCounter_Doc,
    /* tp_traverse       */ nullptr,
    /* tp_clear          */ nullptr,
    /* tp_richcompare    */ nullptr,
    /* tp_weaklistoffset */ 0,
    /* tp_iter           */ nullptr,
    /* tp_iternext       */ nullptr,
    /* tp_methods        */ PyCounterMethods,
    /* tp_members        */ nullptr,
    /* tp_getset         */ nullptr,
    /* tp_base           */ nullptr,
    /* tp_dict           */ nullptr,
    /* tp_descr_get      */ nullptr,
    /* tp_descr_set      */ nullptr,
    /* tp_dictoffset     */ 0,
    /* tp_init           */ nullptr,
    /* tp_alloc          */ nullptr,
    /* tp_new            */ nullptr,
    /* tp_free           */ nullptr,
    /* tp_is_gc          */ nullptr,
    /* tp_bases          */ nullptr,
    /* tp_mro            */ nullptr,
    /* tp_cache          */ nullptr,
    /* tp_subclasses     */ nullptr,
    /* tp_weaklist       */ nullptr,
    /* tp_del            */ nullptr,
    /* tp_version_tag    */ 0,
};

PyTypeObject PyRateType = {
    PyVarObject_HEAD_INIT(&PyType_Type, 0)
    /* tp_name           */ "solomon.Rate",
    /* tp_basicsize      */ sizeof(TPyRate),
    /* tp_itemsize       */ 0,
    /* tp_dealloc        */ TPyRate::Dealloc,
    /* tp_print          */ nullptr,
    /* tp_getattr        */ nullptr,
    /* tp_setattr        */ nullptr,
    /* tp_compare        */ nullptr,
    /* tp_repr           */ TPyRate::Repr,
    /* tp_as_number      */ nullptr,
    /* tp_as_sequence    */ nullptr,
    /* tp_as_mapping     */ nullptr,
    /* tp_hash           */ nullptr,
    /* tp_call           */ nullptr,
    /* tp_str            */ nullptr,
    /* tp_getattro       */ PyObject_GenericGetAttr,
    /* tp_setattro       */ nullptr,
    /* tp_as_buffer      */ nullptr,
    /* tp_flags          */ 0,
    /* tp_doc            */ PyRate_Doc,
    /* tp_traverse       */ nullptr,
    /* tp_clear          */ nullptr,
    /* tp_richcompare    */ nullptr,
    /* tp_weaklistoffset */ 0,
    /* tp_iter           */ nullptr,
    /* tp_iternext       */ nullptr,
    /* tp_methods        */ PyRateMethods,
    /* tp_members        */ nullptr,
    /* tp_getset         */ nullptr,
    /* tp_base           */ nullptr,
    /* tp_dict           */ nullptr,
    /* tp_descr_get      */ nullptr,
    /* tp_descr_set      */ nullptr,
    /* tp_dictoffset     */ 0,
    /* tp_init           */ nullptr,
    /* tp_alloc          */ nullptr,
    /* tp_new            */ nullptr,
    /* tp_free           */ nullptr,
    /* tp_is_gc          */ nullptr,
    /* tp_bases          */ nullptr,
    /* tp_mro            */ nullptr,
    /* tp_cache          */ nullptr,
    /* tp_subclasses     */ nullptr,
    /* tp_weaklist       */ nullptr,
    /* tp_del            */ nullptr,
    /* tp_version_tag    */ 0,
};

PyTypeObject PyHistogramType = {
    PyVarObject_HEAD_INIT(&PyType_Type, 0)
    /* tp_name           */ "solomon.Histogram",
    /* tp_basicsize      */ sizeof(TPyHistogram),
    /* tp_itemsize       */ 0,
    /* tp_dealloc        */ TPyHistogram::Dealloc,
    /* tp_print          */ nullptr,
    /* tp_getattr        */ nullptr,
    /* tp_setattr        */ nullptr,
    /* tp_compare        */ nullptr,
    /* tp_repr           */ TPyHistogram::Repr,
    /* tp_as_number      */ nullptr,
    /* tp_as_sequence    */ nullptr,
    /* tp_as_mapping     */ nullptr,
    /* tp_hash           */ nullptr,
    /* tp_call           */ nullptr,
    /* tp_str            */ nullptr,
    /* tp_getattro       */ PyObject_GenericGetAttr,
    /* tp_setattro       */ nullptr,
    /* tp_as_buffer      */ nullptr,
    /* tp_flags          */ 0,
    /* tp_doc            */ PyHistogram_Doc,
    /* tp_traverse       */ nullptr,
    /* tp_clear          */ nullptr,
    /* tp_richcompare    */ nullptr,
    /* tp_weaklistoffset */ 0,
    /* tp_iter           */ nullptr,
    /* tp_iternext       */ nullptr,
    /* tp_methods        */ PyHistogramMethods,
    /* tp_members        */ nullptr,
    /* tp_getset         */ nullptr,
    /* tp_base           */ nullptr,
    /* tp_dict           */ nullptr,
    /* tp_descr_get      */ nullptr,
    /* tp_descr_set      */ nullptr,
    /* tp_dictoffset     */ 0,
    /* tp_init           */ nullptr,
    /* tp_alloc          */ nullptr,
    /* tp_new            */ nullptr,
    /* tp_free           */ nullptr,
    /* tp_is_gc          */ nullptr,
    /* tp_bases          */ nullptr,
    /* tp_mro            */ nullptr,
    /* tp_cache          */ nullptr,
    /* tp_subclasses     */ nullptr,
    /* tp_weaklist       */ nullptr,
    /* tp_del            */ nullptr,
    /* tp_version_tag    */ 0,
};

NPython2::TObjectPtr WrapGauge(NMonitoring::TGauge* gauge) {
    return TPyGauge::New(gauge);
}

NPython2::TObjectPtr WrapIntGauge(NMonitoring::TIntGauge* gauge) {
    return TPyIntGauge::New(gauge);
}

NPython2::TObjectPtr WrapCounter(NMonitoring::TCounter* counter) {
    return TPyCounter::New(counter);
}

NPython2::TObjectPtr WrapRate(NMonitoring::TRate* rate) {
    return TPyRate::New(rate);
}

NPython2::TObjectPtr WrapHistogram(NMonitoring::THistogram* histogram) {
    return TPyHistogram::New(histogram);
}

} // namespace NAgent
} // namespace NSolomon
