#include "py_cast.h"
#include "py_histogram.h"
#include "py_metric_consumer.h"

#include <solomon/agent/lib/python2/error.h>

#include <library/cpp/monlib/metrics/metric_consumer.h>
#include <library/cpp/monlib/metrics/histogram_snapshot.h>

using namespace NMonitoring;

namespace NSolomon {
namespace NAgent {
namespace {

void ConsumeLabels(IMetricConsumer* consumer, const TLabels* commonLabels, const PyObject* labelsPy) {
    consumer->OnLabelsBegin();
    for (const auto& l: *commonLabels) {
        consumer->OnLabel(TString{l.Name()}, TString{l.Value()});
    }
    PyDictForEach(labelsPy, [consumer](TStringBuf name, TStringBuf value) {
        consumer->OnLabel(TString(name), TString(value));
    });
    consumer->OnLabelsEnd();
}

/////////////////////////////////////////////////////////////////////////////
// TMetricConverter
/////////////////////////////////////////////////////////////////////////////
template <typename DerivedType, EMetricType Type, typename ValueType>
class TMetricConverter {
public:
    TMetricConverter(const TLabels* commonLabels, IMetricConsumer* consumer)
        : CommonLabels_(commonLabels)
        , Consumer_(consumer)
    {
    }

    PyObject* Convert(PyObject* args) {
        PY_TRY {
            PyObject *labelsPy, *tsPy, *valuePy = Py_None;
            const char* funcName = Derived()->FuncName();
            if (!PyArg_UnpackTuple(args, funcName, 2, 3, &labelsPy, &tsPy, &valuePy)) {
                return nullptr;
            }

            Py_ssize_t argsSize = PyTuple_GET_SIZE(args);
            if (argsSize == 3) {
                // received (labels, ts, value)
                ProcessSingleValue(labelsPy, tsPy, valuePy);
            } else {
                // received (labels, [ts1, value1, ts2, value2, ...])
                //
                // the second argument actually an array with an even number of
                // elements, just rename according variable to make it more
                // clear
                //
                PyObject* valuesPy = tsPy;
                ProcessMultipleValues(labelsPy, valuesPy);
            }

            Py_RETURN_NONE;
        } PY_CATCH(nullptr)
    }

protected:
    DerivedType* Derived() noexcept {
        return static_cast<DerivedType*>(this);
    }

    const DerivedType* Derived() const noexcept {
        return static_cast<const DerivedType*>(this);
    }

    void ProcessSingleValue(PyObject* labelsPy, PyObject* tsPy, PyObject* valuePy) {
        // do not add metrics with None values
        if (valuePy != Py_None) {
            // do conversion under acquired GIL
            TInstant ts = PyNumberToInstant(tsPy);
            ValueType value = Derived()->ConvertValue(valuePy);

            Consumer_->OnMetricBegin(Type);
            ConsumeLabels(Consumer_, CommonLabels_, labelsPy);
            Derived()->ConsumePoint(ts, value);
            Consumer_->OnMetricEnd();
        }
    }

    void ProcessMultipleValues(PyObject* labelsPy, PyObject* valuesPy) {
        PY_ENSURE_TYPE(PyList_Check, valuesPy,
                "second argument must be a list");

        Py_ssize_t valuesSize = PyList_GET_SIZE(valuesPy);
        PY_ENSURE(valuesSize % 2 == 0,
                "list with timestamps and values must contain an "
                "even items, but got: " << valuesSize);

        // do not add labels without values
        if (valuesSize != 0) {
            // do conversion under acquired GIL
            std::vector<std::pair<TInstant, ValueType>> values;
            values.reserve(valuesSize);
            for (Py_ssize_t i = 0; i < valuesSize; ) {
                TInstant ts = PyNumberToInstant(PyList_GET_ITEM(valuesPy, i)); ++i;
                ValueType value = Derived()->ConvertValue(PyList_GET_ITEM(valuesPy, i)); ++i;
                values.emplace_back(ts, value);
            }

            Consumer_->OnMetricBegin(Type);
            ConsumeLabels(Consumer_, CommonLabels_, labelsPy);
            for (const auto& it: values) {
                Derived()->ConsumePoint(it.first, it.second);
            }
            Consumer_->OnMetricEnd();
        }
    }

protected:
    const TLabels* CommonLabels_;
    IMetricConsumer* Consumer_;
};

/////////////////////////////////////////////////////////////////////////////
// TGaugeConverter
/////////////////////////////////////////////////////////////////////////////
class TGaugeConverter: public TMetricConverter<TGaugeConverter, EMetricType::GAUGE, double> {
    using TBase = TMetricConverter<TGaugeConverter, EMetricType::GAUGE, double>;
public:
    TGaugeConverter(const TLabels* commonLabels, IMetricConsumer* consumer)
        : TBase(commonLabels, consumer)
    {
    }

    const char* FuncName() const noexcept {
        return "gauge";
    }

    double ConvertValue(PyObject* valuePy) {
        return PyNumberToDouble(valuePy);
    }

    void ConsumePoint(TInstant ts, double value) {
        Consumer_->OnDouble(ts, value);
    }
};

/////////////////////////////////////////////////////////////////////////////
// TIntGaugeConverter
/////////////////////////////////////////////////////////////////////////////
class TIntGaugeConverter: public TMetricConverter<TIntGaugeConverter, EMetricType::IGAUGE, i64> {
    using TBase = TMetricConverter<TIntGaugeConverter, EMetricType::IGAUGE, i64>;
public:
    TIntGaugeConverter(const TLabels* commonLabels, IMetricConsumer* consumer)
        : TBase(commonLabels, consumer)
    {
    }

    const char* FuncName() const noexcept {
        return "igauge";
    }

    i64 ConvertValue(PyObject* valuePy) {
        return PyNumberToLong(valuePy);
    }

    void ConsumePoint(TInstant ts, i64 value) {
        Consumer_->OnInt64(ts, value);
    }
};

/////////////////////////////////////////////////////////////////////////////
// TCounterConverter
/////////////////////////////////////////////////////////////////////////////
class TCounterConverter: public TMetricConverter<TCounterConverter, EMetricType::COUNTER, ui64> {
    using TBase = TMetricConverter<TCounterConverter, EMetricType::COUNTER, ui64>;
public:
    TCounterConverter(const TLabels* commonLabels, IMetricConsumer* consumer)
        : TBase(commonLabels, consumer)
    {
    }

    const char* FuncName() const noexcept {
        return "counter";
    }

    ui64 ConvertValue(PyObject* valuePy) {
        return PyNumberToUnsignedLong(valuePy);
    }

    void ConsumePoint(TInstant ts, ui64 value) {
        Consumer_->OnUint64(ts, value);
    }
};

/////////////////////////////////////////////////////////////////////////////
// TRateConverter
/////////////////////////////////////////////////////////////////////////////
class TRateConverter: public TMetricConverter<TRateConverter, EMetricType::RATE, ui64> {
    using TBase = TMetricConverter<TRateConverter, EMetricType::RATE, ui64>;
public:
    TRateConverter(const TLabels* commonLabels, IMetricConsumer* consumer)
        : TBase(commonLabels, consumer)
    {
    }

    const char* FuncName() const noexcept {
        return "rate";
    }

    ui64 ConvertValue(PyObject* valuePy) {
        return PyNumberToUnsignedLong(valuePy);
    }

    void ConsumePoint(TInstant ts, ui64 value) {
        Consumer_->OnUint64(ts, value);
    }
};

template <typename TDerived, EMetricType MetricType>
class THistConverterBase {
public:
    THistConverterBase(const TLabels* commonLabels, IMetricConsumer* consumer)
        : CommonLabels_{commonLabels}
        , Consumer_{consumer}
    {
    }

    PyObject* Convert(PyObject* args) {
        PY_TRY {
            PyObject *labelsPy, *tsPy, *bucketsPy, *valuesPy = Py_None;
            const char* funcName = Derived()->FuncName();
            if (!PyArg_UnpackTuple(args, funcName, 4, 4, &labelsPy, &tsPy, &bucketsPy, &valuesPy)) {
                return nullptr;
            }

            PY_ENSURE((valuesPy != Py_None && bucketsPy != Py_None), "Bucket and value lists must be not None");
            TInstant ts = PyNumberToInstant(tsPy);
            TBucketBounds buckets = PyListToVectorFloat(bucketsPy);
            TBucketValues values = PyListToVectorUi64(valuesPy);

            PY_ENSURE(buckets.size() >= 1, "Histogram must have at least one bucket");
            PY_ENSURE(IsSorted(buckets.begin(), buckets.end()), "Buckets must be sorted");
            PY_ENSURE(buckets.size() < HISTOGRAM_MAX_BUCKETS_COUNT, "Bucket count must be less than "
                << HISTOGRAM_MAX_BUCKETS_COUNT << ", but is " << buckets.size());
            PY_ENSURE((buckets.size() == values.size() || buckets.size() == values.size() - 1),
                "Size of value list must be equal to bucket list size or greater than it by 1");

            buckets.push_back(Max<TBucketBound>());

            if (values.size() != buckets.size()) {
                values.push_back(0);
            }

            IHistogramSnapshotPtr snapshot = ExplicitHistogramSnapshot(buckets, values);

            Consumer_->OnMetricBegin(MetricType);
            ConsumeLabels(Consumer_, CommonLabels_, labelsPy);
            Consumer_->OnHistogram(ts, std::move(snapshot));
            Consumer_->OnMetricEnd();

            Py_RETURN_NONE;
        } PY_CATCH(nullptr)
    }

protected:
    TDerived* Derived() noexcept {
        return static_cast<TDerived*>(this);
    }

    const TDerived* Derived() const noexcept {
        return static_cast<const TDerived*>(this);
    }

private:
    const TLabels* CommonLabels_;
    IMetricConsumer* Consumer_;
};

struct THistRateConverter: THistConverterBase<THistRateConverter, EMetricType::HIST_RATE> {
    THistRateConverter(const TLabels* commonLabels, IMetricConsumer* consumer)
        : THistConverterBase<THistRateConverter, EMetricType::HIST_RATE>{commonLabels, consumer}
    {
    }

    const char* FuncName() {
        return "histogram_rate";
    }
};

struct THistCounterConverter: THistConverterBase<THistCounterConverter, EMetricType::HIST> {
    THistCounterConverter(const TLabels* commonLabels, IMetricConsumer* consumer)
        : THistConverterBase<THistCounterConverter, EMetricType::HIST>{commonLabels, consumer}
    {
    }

    const char* FuncName() {
        return "histogram_counter";
    }
};

/////////////////////////////////////////////////////////////////////////////
// TPyMetricConsumer
/////////////////////////////////////////////////////////////////////////////
struct TPyMetricConsumer {
    PyObject_HEAD
    const TLabels* CommonLabels;
    IMetricConsumer* Delegate;

    static TPyMetricConsumer* Cast(PyObject* o) {
        return reinterpret_cast<TPyMetricConsumer*>(o);
    }

    static void Dealloc(PyObject* self) {
        delete Cast(self);
    }

    static PyObject* New(
            const TLabels* commonLabels,
            IMetricConsumer* delegate)
    {
        TPyMetricConsumer* consumer = new TPyMetricConsumer;
        PyObject_INIT(consumer, &PyMetricConsumerType);
        consumer->CommonLabels = commonLabels;
        consumer->Delegate = delegate;
        return reinterpret_cast<PyObject*>(consumer);
    }

    static PyObject* Repr(PyObject* self) {
        return PyString_FromFormat("<solomon.MetricConsumer object at %p>", self);
    }

    static PyObject* Gauge(PyObject* self, PyObject* args) {
        TPyMetricConsumer* me = Cast(self);
        TGaugeConverter c(me->CommonLabels, me->Delegate);
        return c.Convert(args);
    }

    static PyObject* IntGauge(PyObject* self, PyObject* args) {
        TPyMetricConsumer* me = Cast(self);
        TIntGaugeConverter c(me->CommonLabels, me->Delegate);
        return c.Convert(args);
    }

    static PyObject* Counter(PyObject* self, PyObject* args) {
        TPyMetricConsumer* me = Cast(self);
        TCounterConverter c(me->CommonLabels, me->Delegate);
        return c.Convert(args);
    }

    static PyObject* Rate(PyObject* self, PyObject* args) {
        TPyMetricConsumer* me = Cast(self);
        TRateConverter c(me->CommonLabels, me->Delegate);
        return c.Convert(args);
    }

    static PyObject* HistogramRate(PyObject* self, PyObject* args) {
        TPyMetricConsumer* me = Cast(self);
        THistRateConverter c(me->CommonLabels, me->Delegate);
        return c.Convert(args);
    }

    static PyObject* HistogramCounter(PyObject* self, PyObject* args) {
        TPyMetricConsumer* me = Cast(self);
        THistCounterConverter c(me->CommonLabels, me->Delegate);
        return c.Convert(args);
    }
};

PyDoc_STRVAR(PyMetricConsumer_Doc,
    "TODO: add solomon.MetricConsumer docs");
PyDoc_STRVAR(PyGauge_Doc,
    "TODO: add solomon.MetricConsumer::gauge(labels, ts, value) docs\n"
    "TODO: add solomon.MetricConsumer::gauge(labels, ts_and_values) docs\n");
PyDoc_STRVAR(PyIntGauge_Doc,
    "TODO: add solomon.MetricConsumer::igauge(labels, ts, value) docs\n"
    "TODO: add solomon.MetricConsumer::igauge(labels, ts_and_values) docs\n");
PyDoc_STRVAR(PyCounter_Doc,
    "TODO: add solomon.MetricConsumer::counter(labels, ts, value) docs\n"
    "TODO: add solomon.MetricConsumer::counter(labels, ts_and_values) docs\n");
PyDoc_STRVAR(PyRate_Doc,
    "TODO: add solomon.MetricConsumer::rate(labels, ts, value) docs\n"
    "TODO: add solomon.MetricConsumer::rate(labels, ts_and_values) docs\n");
PyDoc_STRVAR(PyHistogram_Doc,
    "TODO: add solomon.MetricConsumer::rate(labels, ts, value) docs\n"
    "TODO: add solomon.MetricConsumer::rate(labels, ts_and_values) docs\n");

static PyMethodDef PyMetricConsumerMethods[] = {
    { "gauge",  TPyMetricConsumer::Gauge, METH_VARARGS, PyGauge_Doc },
    { "igauge",  TPyMetricConsumer::IntGauge, METH_VARARGS, PyIntGauge_Doc },
    { "counter",  TPyMetricConsumer::Counter, METH_VARARGS, PyCounter_Doc },
    { "rate",  TPyMetricConsumer::Rate, METH_VARARGS, PyRate_Doc },
    { "histogram_counter",  TPyMetricConsumer::HistogramCounter, METH_VARARGS, PyHistogram_Doc },
    { "histogram_rate",  TPyMetricConsumer::HistogramRate, METH_VARARGS, PyHistogram_Doc },
    { nullptr, nullptr, 0, nullptr }    /* sentinel */
};

} // namespace

PyTypeObject PyMetricConsumerType = {
    PyVarObject_HEAD_INIT(&PyType_Type, 0)
    /* tp_name           */ "solomon.MetricConsumer",
    /* tp_basicsize      */ sizeof(TPyMetricConsumer),
    /* tp_itemsize       */ 0,
    /* tp_dealloc        */ TPyMetricConsumer::Dealloc,
    /* tp_print          */ nullptr,
    /* tp_getattr        */ nullptr,
    /* tp_setattr        */ nullptr,
    /* tp_compare        */ nullptr,
    /* tp_repr           */ TPyMetricConsumer::Repr,
    /* tp_as_number      */ nullptr,
    /* tp_as_sequence    */ nullptr,
    /* tp_as_mapping     */ nullptr,
    /* tp_hash           */ nullptr,
    /* tp_call           */ nullptr,
    /* tp_str            */ nullptr,
    /* tp_getattro       */ PyObject_GenericGetAttr,
    /* tp_setattro       */ nullptr,
    /* tp_as_buffer      */ nullptr,
    /* tp_flags          */ 0,
    /* tp_doc            */ PyMetricConsumer_Doc,
    /* tp_traverse       */ nullptr,
    /* tp_clear          */ nullptr,
    /* tp_richcompare    */ nullptr,
    /* tp_weaklistoffset */ 0,
    /* tp_iter           */ nullptr,
    /* tp_iternext       */ nullptr,
    /* tp_methods        */ PyMetricConsumerMethods,
    /* tp_members        */ nullptr,
    /* tp_getset         */ nullptr,
    /* tp_base           */ nullptr,
    /* tp_dict           */ nullptr,
    /* tp_descr_get      */ nullptr,
    /* tp_descr_set      */ nullptr,
    /* tp_dictoffset     */ 0,
    /* tp_init           */ nullptr,
    /* tp_alloc          */ nullptr,
    /* tp_new            */ nullptr,
    /* tp_free           */ nullptr,
    /* tp_is_gc          */ nullptr,
    /* tp_bases          */ nullptr,
    /* tp_mro            */ nullptr,
    /* tp_cache          */ nullptr,
    /* tp_subclasses     */ nullptr,
    /* tp_weaklist       */ nullptr,
    /* tp_del            */ nullptr,
    /* tp_version_tag    */ 0,
};

NPython2::TObjectPtr WrapMetricConsumer(
        const TLabels* commonLabels,
        IMetricConsumer* consumer)
{
    return TPyMetricConsumer::New(commonLabels, consumer);
}


} // namespace NAgent
} // namespace NSolomon
