#include "py_cast.h"
#include "py_histogram.h"
#include "py_metric.h"
#include "py_metric_registry.h"

#include <solomon/agent/lib/python2/error.h>
#include <solomon/agent/lib/python2/py_string.h>

#include <library/cpp/monlib/metrics/metric_registry.h>
#include <library/cpp/monlib/metrics/histogram_collector.h>

namespace NSolomon {
namespace NAgent {
namespace {

using namespace NMonitoring;

/////////////////////////////////////////////////////////////////////////////
// TPyMetricRegistry
/////////////////////////////////////////////////////////////////////////////
struct TPyMetricRegistry {
    PyObject_HEAD
    NMonitoring::TMetricRegistry* Delegate;

    static TPyMetricRegistry* Cast(PyObject* o) {
        return reinterpret_cast<TPyMetricRegistry*>(o);
    }

    static void Dealloc(PyObject* self) {
        delete Cast(self);
    }

    static PyObject* New(NMonitoring::TMetricRegistry* delegate) {
        TPyMetricRegistry* registry = new TPyMetricRegistry;
        PyObject_INIT(registry, &PyMetricRegistryType);
        registry->Delegate = delegate;
        return reinterpret_cast<PyObject*>(registry);
    }

    static PyObject* Repr(PyObject* self) {
        return PyString_FromFormat("<solomon.MetricRegistry object at %p>", self);
    }

    template <typename TMetric>
    static PyObject* PlainMetricFactory(PyObject* self, PyObject* labelsPy,
        TMetric* (TMetricRegistry::*factoryMethod)(NMonitoring::TLabels),
        NPython2::TObjectPtr (*wrapper)(TMetric*)
    )
    {
        PY_TRY {
            NMonitoring::TLabels labels;
            PyDictToLabels(labelsPy, &labels);

            auto* sensor = (Cast(self)->Delegate->*factoryMethod)(labels);
            return wrapper(sensor).Release();
        } PY_CATCH(nullptr)
    }

    static PyObject* Gauge(PyObject* self, PyObject* labelsPy) {
        return PlainMetricFactory(self, labelsPy, &TMetricRegistry::Gauge, WrapGauge);
    }

    static PyObject* IntGauge(PyObject* self, PyObject* labelsPy) {
        return PlainMetricFactory(self, labelsPy, &TMetricRegistry::IntGauge, WrapIntGauge);
    }

    static PyObject* Counter(PyObject* self, PyObject* labelsPy) {
        return PlainMetricFactory(self, labelsPy, &TMetricRegistry::Counter, WrapCounter);
    }

    static PyObject* Rate(PyObject* self, PyObject* labelsPy) {
        return PlainMetricFactory(self, labelsPy, &TMetricRegistry::Rate, WrapRate);
    }

    using THistogramFactoryMethod = THistogram* (TMetricRegistry::*)(NMonitoring::TLabels, IHistogramCollectorPtr);
    static PyObject* HistogramFactoryImpl(THistogramFactoryMethod method, PyObject* self, PyObject* args, PyObject* kwargs) {
        PY_TRY {
            NMonitoring::TLabels labels;
            auto collector = CreateHistCollectorFromPyArgs(args, kwargs, &labels);
            auto* histogram = (Cast(self)->Delegate->*method)(labels, std::move(collector));
            return WrapHistogram(histogram).Release();
        } PY_CATCH(nullptr)
    }

    static PyObject* HistogramRate(PyObject* self, PyObject* args, PyObject* kwargs) {
        return HistogramFactoryImpl(&TMetricRegistry::HistogramRate, self, args, kwargs);
    }

    static PyObject* HistogramCounter(PyObject* self, PyObject* args, PyObject* kwargs) {
        return HistogramFactoryImpl(&TMetricRegistry::HistogramCounter, self, args, kwargs);
    }
};

PyDoc_STRVAR(PyMetricRegistry_Doc,
    "TODO: add solomon.MetricRegistry docs");
PyDoc_STRVAR(PyGauge_Doc,
    "TODO: add solomon.MetricRegistry::gauge(labels) docs");
PyDoc_STRVAR(PyCounter_Doc,
    "TODO: add solomon.MetricRegistry::counter(labels) docs");
PyDoc_STRVAR(PyRate_Doc,
    "TODO: add solomon.MetricRegistry::rate(labels) docs");
PyDoc_STRVAR(PyHistogramRate_Doc,
    "Creates a new instance of rate histogram");
PyDoc_STRVAR(PyHistogramCounter_Doc,
    "Creates a new instance of counter histogram");

static PyMethodDef PyMetricRegistryMethods[] = {
    { "create_gauge",  TPyMetricRegistry::Gauge, METH_O, PyGauge_Doc }, // DEPRECATED
    { "gauge",  TPyMetricRegistry::Gauge, METH_O, PyGauge_Doc },
    { "igauge",  TPyMetricRegistry::IntGauge, METH_O, PyGauge_Doc },
    { "create_counter",  TPyMetricRegistry::Counter, METH_O, PyCounter_Doc },  // DEPRECATED
    { "counter",  TPyMetricRegistry::Counter, METH_O, PyCounter_Doc },
    { "create_rate",  TPyMetricRegistry::Rate, METH_O, PyRate_Doc },  // DEPRECATED
    { "rate",  TPyMetricRegistry::Rate, METH_O, PyRate_Doc },
    { "histogram_counter",  reinterpret_cast<PyCFunction>(TPyMetricRegistry::HistogramCounter), METH_KEYWORDS|METH_VARARGS, PyHistogramCounter_Doc },
    { "histogram_rate",  reinterpret_cast<PyCFunction>(TPyMetricRegistry::HistogramRate), METH_KEYWORDS|METH_VARARGS, PyHistogramRate_Doc },
    { nullptr, nullptr, 0, nullptr }    /* sentinel */
};

} // namespace

PyTypeObject PyMetricRegistryType = {
    PyVarObject_HEAD_INIT(&PyType_Type, 0)
    /* tp_name           */ "solomon.MetricRegistry",
    /* tp_basicsize      */ sizeof(TPyMetricRegistry),
    /* tp_itemsize       */ 0,
    /* tp_dealloc        */ TPyMetricRegistry::Dealloc,
    /* tp_print          */ nullptr,
    /* tp_getattr        */ nullptr,
    /* tp_setattr        */ nullptr,
    /* tp_compare        */ nullptr,
    /* tp_repr           */ TPyMetricRegistry::Repr,
    /* tp_as_number      */ nullptr,
    /* tp_as_sequence    */ nullptr,
    /* tp_as_mapping     */ nullptr,
    /* tp_hash           */ nullptr,
    /* tp_call           */ nullptr,
    /* tp_str            */ nullptr,
    /* tp_getattro       */ PyObject_GenericGetAttr,
    /* tp_setattro       */ nullptr,
    /* tp_as_buffer      */ nullptr,
    /* tp_flags          */ 0,
    /* tp_doc            */ PyMetricRegistry_Doc,
    /* tp_traverse       */ nullptr,
    /* tp_clear          */ nullptr,
    /* tp_richcompare    */ nullptr,
    /* tp_weaklistoffset */ 0,
    /* tp_iter           */ nullptr,
    /* tp_iternext       */ nullptr,
    /* tp_methods        */ PyMetricRegistryMethods,
    /* tp_members        */ nullptr,
    /* tp_getset         */ nullptr,
    /* tp_base           */ nullptr,
    /* tp_dict           */ nullptr,
    /* tp_descr_get      */ nullptr,
    /* tp_descr_set      */ nullptr,
    /* tp_dictoffset     */ 0,
    /* tp_init           */ nullptr,
    /* tp_alloc          */ nullptr,
    /* tp_new            */ nullptr,
    /* tp_free           */ nullptr,
    /* tp_is_gc          */ nullptr,
    /* tp_bases          */ nullptr,
    /* tp_mro            */ nullptr,
    /* tp_cache          */ nullptr,
    /* tp_subclasses     */ nullptr,
    /* tp_weaklist       */ nullptr,
    /* tp_del            */ nullptr,
    /* tp_version_tag    */ 0,
};

NPython2::TObjectPtr WrapMetricRegistry(NMonitoring::TMetricRegistry* registry) {
    Y_VERIFY(registry);
    return TPyMetricRegistry::New(registry);
}

} // namespace NAgent
} // namespace NSolomon
