#include "py_cast.h"
#include "py_metric_consumer.h"
#include "py_metric_registry.h"
#include "python2.h"

#include <solomon/agent/lib/python2/code_module.h>
#include <solomon/agent/lib/python2/error.h>
#include <solomon/agent/lib/python2/py_string.h>
#include <solomon/agent/lib/python2/gil.h>
#include <solomon/agent/lib/python2/logger.h>
#include <solomon/agent/lib/python2/types.h>
#include <solomon/agent/modules/pull/python2/protos/python2_config.pb.h>
#include <solomon/agent/protos/python2_config.pb.h>

#include <library/cpp/monlib/metrics/metric_registry.h>

#include <util/string/builder.h>

namespace NSolomon {
namespace NAgent {
namespace {

TString MakeAgentModuleName(const TPullPython2Config& config) {
    TStringBuilder ss;
    ss << config.GetModuleName() << "." << config.GetClassName();

    if (config.ParamsSize() > 0) {
        auto&& params = config.GetParams();

        TVector<std::pair<TStringBuf, TStringBuf>> paramsList;
        for (auto pair: params) {
            paramsList.emplace_back(pair);
        }

        Sort(paramsList);

        for (const auto& it: paramsList) {
            ss << "-" << it.first << "_" << it.second;
        }
    }

    return ss;
}

class TPullPython2Module: public IPullModule {
public:
    TPullPython2Module(
            const TLabels& labels,
            const TPullPython2Config& config,
            const TGlobalPython2Config& pyConfig)
        : ClassName_(config.GetClassName())
        , AgentModuleName_(MakeAgentModuleName(config))
        , CommonLabels_(labels)
    {
        NPython2::TGilGuard gil;

        auto moduleCode = NPython2::LoadPyModule(
                    config.GetModuleName(),
                    config.GetFilePath(),
                    pyConfig.GetIgnorePycFiles());

        // (1) check class
        NPython2::TObjectPtr moduleClass = PyObject_GetAttrString(
                moduleCode.Get(), ClassName_.c_str());
        PY_ENSURE(moduleClass,
                "cannot find class '" << ClassName_
                << "' in " << config.GetFilePath());
        PY_ENSURE_TYPE(NPython2::IsPythonClass, moduleClass.Get(),
                "expected to get a class by name '" << ClassName_
                << "' in " << config.GetFilePath());

        // (2) create an instance
        NPython2::TObjectPtr initArgs = PyTuple_New(2);
        PyTuple_SET_ITEM(initArgs.Get(), 0, NPython2::GetLogger().Release());
        PyTuple_SET_ITEM(initArgs.Get(), 1,
                WrapMetricRegistry(&MetricRegistry_).Release());

        NPython2::TObjectPtr initKwArgs;
        if (config.ParamsSize() > 0) {
            initKwArgs.ResetSteal(PyDict_New());
            for (const auto& it: config.GetParams()) {
                NPython2::TObjectPtr key = NPython2::ToPyString(it.first);
                NPython2::TObjectPtr value = NPython2::ToPyString(it.second);
                int rc = PyDict_SetItem(initKwArgs.Get(), key.Get(), value.Get());
                PY_ENSURE(rc == 0, "cannot insert item into init kwargs for " << ClassName_);
            }
        }

        PyModule_ = PyObject_Call(moduleClass.Get(), initArgs.Get(), initKwArgs.Get());
        PY_ENSURE(PyModule_,
                "cannot create instance of " << ClassName_ << " class"
                 <<", python error: " << NPython2::LastErrorAsString());

        // (3) check pull() method
        PyPullMethod_ = PyObject_GetAttrString(PyModule_.Get(), "pull");
        PY_ENSURE(PyPullMethod_,
                "cannot find method pull() in class " << ClassName_);
        PY_ENSURE(PyMethod_Check(PyPullMethod_.Get()),
                "expected to get a method pull() but class " << ClassName_
                << " contains " << NPython2::TypeName(PyPullMethod_.Get())
                << " with this name");
    }

    ~TPullPython2Module() {
        // predictable destroy python objects under GIL
        NPython2::TGilGuard gil;
        PyPullMethod_.Reset();
        PyModule_.Reset();
    }

private:
    TStringBuf Name() const override {
        return AgentModuleName_;
    }

    int Pull(TInstant time, NMonitoring::IMetricConsumer* consumer) override {
        // TODO: must be moved to the upper layer
        // -- cut -------------------------------------------
        consumer->OnLabelsBegin();
        for (const auto& label: CommonLabels_) {
            consumer->OnLabel(TString{label.Name()}, TString{label.Value()});
        }
        consumer->OnLabelsEnd();
        // -- cut -------------------------------------------

        int retcode;
        {
            NPython2::TGilGuard gil;
            TLabels emptyLabels;

            auto pyConsumer = WrapMetricConsumer(&emptyLabels, consumer);

            NPython2::TObjectPtr args = PyTuple_New(2);
            PyTuple_SET_ITEM(args.Get(), 0, InstantToPyNumber(time));
            PyTuple_SET_ITEM(args.Get(), 1, pyConsumer.Release());

            NPython2::TObjectPtr res = PyObject_Call(
                    PyPullMethod_.Get(), args.Get(), nullptr);

            PY_ENSURE(res,
                "cannot pull from module " << Name() << ": "
                << NPython2::LastErrorAsString());

            auto* o = res.Get();

            if (o == Py_None) {
                retcode = 0;
            // bool is checked separately because we don't want to allow implicit conversion
            // to number in order to prevent unexpected behavior
            } else if (PyBool_Check(o)) {
                PY_FAIL("pull() method of " << Name() << " must return None or a number, but got: "
                    << NPython2::TypeName(res.Get()));
            } else if (PyNumber_Check(o)) {
                NPython2::TObjectPtr pyLong = PyNumber_Long(o);
                PY_ENSURE(pyLong, "failed to convert value to long");
                retcode = PyLong_AsLong(pyLong.Get());
                PY_ENSURE(!PyErr_Occurred(), "failed to parse long from PyLong");
            } else {
                PY_FAIL("pull() method of " << Name() << " must return None or a number, but got: "
                    << NPython2::TypeName(res.Get()));
            }

        }

        // correct ts values will be set later, in DataPuller. For now just use Zero()
        MetricRegistry_.Accept(TInstant::Zero(), consumer);
        return retcode;
    }

private:
    const TString ClassName_;
    const TString AgentModuleName_;
    const TLabels CommonLabels_;
    NMonitoring::TMetricRegistry MetricRegistry_;
    NPython2::TObjectPtr PyModule_;
    NPython2::TObjectPtr PyPullMethod_;
};

} // namespace


IPullModulePtr CreatePython2Module(
        const TLabels& labels,
        const TPullPython2Config& config,
        const TGlobalPython2Config& pyConfig)
{
    return new TPullPython2Module(labels, config, pyConfig);
}

} // namespace NAgent
} // namespace NSolomon
