#include <solomon/agent/modules/pull/python2/python2.h>

#include <solomon/agent/modules/pull/python2/protos/python2_config.pb.h>
#include <solomon/agent/protos/python2_config.pb.h>

#include <library/cpp/monlib/metrics/metric.h>
#include <library/cpp/monlib/encode/protobuf/protobuf.h>
#include <solomon/agent/lib/python2/initializer.h>
#include <solomon/agent/lib/python2/error.h>

#include <library/cpp/monlib/encode/protobuf/protos/samples.pb.h>
#include <library/cpp/testing/gtest/gtest.h>

#include <util/system/tempfile.h>

using namespace NMonitoring;
using namespace NSolomon;
using namespace NAgent;

TGlobalPython2Config PyConfig;
NPython2::TInitializer*Initializer = NPython2::TInitializer::Instance();

TString CANCELLING_TEST_PY_MODULE = R"(
class MyPullModule:
    def __init__(self, logger, registry, **params):
        pass

    def pull(self, ts, consumer):
        return )";

TStringBuf TEST_PY_MODULE = R"(
class MyPullModule:
    def __init__(self, logger, registry, p1='1', p2='2', **params):
        self._reqs = registry.counter({'requests': 'count'})
        self._users = registry.gauge({'users': 'count'})
        self._users_int = registry.igauge({'usersInt': 'count'})
        self._bytes_rx = registry.rate({'sensor': 'bytes_rx'})
        assert p1 == '111'
        assert p2 == '2'
        assert 'p3' in params
        assert params['p3'] == '333'
        self._hist_rate = registry.histogram_rate(
            {'sensor': 'hist_rate_explicit'},
            'explicit',
            buckets=[10, 20, 50],
            bucket_count=100)

        self._hist_counter = registry.histogram_counter(
            {'sensor': 'hist_counter_linear'},
            'linear',
            bucket_count=6, start_value=5, bucket_width=15)

    def pull(self, ts, consumer):
        # (1) sensors from registry
        self._reqs.inc()
        self._users.set(10)
        self._users_int.set(100)
        self._bytes_rx.add(20)

        # (2) dynamic sensors with single value
        consumer.counter({'my': 'counter1'}, ts, 1)
        consumer.gauge({'my': 'gauge1'}, ts, 2.3)
        consumer.igauge({'my': 'int_gauge'}, ts, 9223372036854775807)
        consumer.rate({'my': 'rate1'}, ts, 4)

        # (3) dynamic sensors with multiple values
        consumer.counter({'my': 'counter2'}, [
            10, 1,
            20, 2,
            30, 3,
        ])
        consumer.gauge({'my': 'gauge2'}, [
            10, 2.3,
            20, 4.5,
            30, 6.7,
        ])
        consumer.rate({'my': 'rate2'}, [
            10, 2,
            20, 4,
            30, 8,
        ])

        # (4) None must be skipped
        consumer.counter({'my': 'counter3'}, ts, None)
        consumer.gauge({'my': 'gauge3'}, ts, None)
        consumer.rate({'my': 'rate3'}, ts, None)

        # (5) Write hist via consumer
        consumer.histogram_counter(
            {'consumer': 'hist_counter_explicit'}, 0, [0, 5],
            [10, 20, 50])

        consumer.histogram_rate(
            {'consumer': 'hist_rate_exp'}, 42, [25, 50], [2, 3])

)";

void CheckPoint(const NProto::TPoint& p, ui64 ts, ui64 value) {
    ASSERT_EQ(p.GetTime(), ts);
    ASSERT_EQ(p.GetUint64(), value);
}

void CheckPoint(const NProto::TPoint& p, ui64 ts, i64 value) {
    ASSERT_EQ(p.GetTime(), ts);
    ASSERT_EQ(p.GetInt64(), value);
}

void CheckPoint(const NProto::TPoint& p, ui64 ts, double value) {
    ASSERT_EQ(p.GetTime(), ts);
    ASSERT_DOUBLE_EQ(p.GetFloat64(), value);
}

void ExpectHistogram(const NProto::THistogram& hist, const TBucketBounds& expectedBuckets, const TBucketValues& expectedValues) {
    ASSERT_EQ(hist.BoundsSize(), expectedBuckets.size());
    for (size_t i = 0; i < hist.BoundsSize(); ++i) {
        ASSERT_EQ(expectedBuckets[i], hist.GetBounds(i));
    }

    ASSERT_EQ(hist.ValuesSize(), expectedValues.size());
    for (size_t i = 0; i < hist.ValuesSize(); ++i) {
        ASSERT_EQ(expectedValues[i], hist.GetValues(i));
    }
}

struct TPyModule : TMoveOnly {
    TPyModule(TStringBuf text, const TAgentLabels& labels) {
        File.Write(text.data(), text.size());
        File.Flush();

        TPullPython2Config config;
        config.SetFilePath(File.Name());
        config.SetModuleName("test");
        config.SetClassName("MyPullModule");

        auto* params = config.MutableParams();
        params->insert({ "p1", "111"});
        params->insert({ "p3", "333"});

        Module = CreatePython2Module(labels, config, PyConfig);
    }

    TTempFileHandle File;
    TPullPython2Config Config;
    IPullModulePtr Module;
};

TEST(TPython2PullTest, PullModuleReturnValue) {
    NProto::TMultiSamplesList samples;
    auto encoder = EncoderProtobuf(&samples);
    auto now = TInstant::Now();

    {
        const auto text = CANCELLING_TEST_PY_MODULE + "1";
        TPyModule holder{text, {}};
        holder.Module->Pull(now, encoder.Get());
    }

    {
        const auto text = CANCELLING_TEST_PY_MODULE + "False";
        TPyModule holder{text, {}};
        ASSERT_THROW(holder.Module->Pull(now, encoder.Get()), yexception);
    }

    {
        const auto text = CANCELLING_TEST_PY_MODULE + "0L";
        TPyModule holder{text, {}};
        holder.Module->Pull(now, encoder.Get());
    }

    {
        const auto text = CANCELLING_TEST_PY_MODULE + "1.2";
        TPyModule holder{text, {}};
        holder.Module->Pull(now, encoder.Get());
    }

    {
        const auto text = CANCELLING_TEST_PY_MODULE + "None";
        TPyModule holder{text, {}};
        holder.Module->Pull(now, encoder.Get());
    }

    {
        const auto text = CANCELLING_TEST_PY_MODULE + "\"foo\"";
        TPyModule holder{text, {}};
        ASSERT_THROW(holder.Module->Pull(now, encoder.Get()), yexception);
    }
}

TEST(TPython2PullTest, Pull) {
    TAgentLabels commonLabels = { {"os", "macOS"}, {"kernel", "Darwin"} };
    TPyModule holder{TEST_PY_MODULE, commonLabels};
    auto& module = holder.Module;

    NProto::TMultiSamplesList samples;
    auto encoder = EncoderProtobuf(&samples);
    auto now = TInstant::Now();
    auto nowTs = now.MilliSeconds();
    ui64 registryTs = 0; // because metrics created from a registry don't have ts (see SOLOMON-4830)

    module->Pull(now, encoder.Get());

    ASSERT_EQ(samples.CommonLabelsSize(), 2u);
    {
        auto label1 = samples.GetCommonLabels(0);
        ASSERT_EQ(label1.GetName(), "os");
        ASSERT_EQ(label1.GetValue(), "macOS");

        auto label2 = samples.GetCommonLabels(1);
        ASSERT_EQ(label2.GetName(), "kernel");
        ASSERT_EQ(label2.GetValue(), "Darwin");
    }

    ASSERT_EQ(samples.SamplesSize(), 15u);
    for (const NProto::TMultiSample& sample: samples.GetSamples()) {
        auto& label0 = sample.GetLabels(0);
        if (label0.GetName() == "my") {
            if (label0.GetValue() == "counter1") {
                ASSERT_EQ(sample.GetMetricType(), NProto::COUNTER);
                ASSERT_EQ(sample.PointsSize(), 1u);
                CheckPoint(sample.GetPoints(0), nowTs, ui64(1));
            } else if (label0.GetValue() == "counter2") {
                ASSERT_EQ(sample.GetMetricType(), NProto::COUNTER);
                ASSERT_EQ(sample.PointsSize(), 3u);
                CheckPoint(sample.GetPoints(0), 10, ui64(1));
                CheckPoint(sample.GetPoints(1), 20, ui64(2));
                CheckPoint(sample.GetPoints(2), 30, ui64(3));
            } else if (label0.GetValue() == "gauge1") {
                ASSERT_EQ(sample.GetMetricType(), NProto::GAUGE);
                ASSERT_EQ(sample.PointsSize(), 1u);
                CheckPoint(sample.GetPoints(0), nowTs, 2.3);
            } else if (label0.GetValue() == "int_gauge") {
                ASSERT_EQ(sample.GetMetricType(), NProto::IGAUGE);
                ASSERT_EQ(sample.PointsSize(), 1u);
                CheckPoint(sample.GetPoints(0), nowTs, Max<i64>());
            } else if (label0.GetValue() == "gauge2") {
                ASSERT_EQ(sample.GetMetricType(), NProto::GAUGE);
                ASSERT_EQ(sample.PointsSize(), 3u);
                CheckPoint(sample.GetPoints(0), 10, 2.3);
                CheckPoint(sample.GetPoints(1), 20, 4.5);
                CheckPoint(sample.GetPoints(2), 30, 6.7);
            } else if (label0.GetValue() == "rate1") {
                ASSERT_EQ(sample.GetMetricType(), NProto::RATE);
                ASSERT_EQ(sample.PointsSize(), 1u);
                CheckPoint(sample.GetPoints(0), nowTs, ui64(4));
            } else if (label0.GetValue() == "rate2") {
                ASSERT_EQ(sample.GetMetricType(), NProto::RATE);
                ASSERT_EQ(sample.PointsSize(), 3u);
                CheckPoint(sample.GetPoints(0), 10, ui64(2));
                CheckPoint(sample.GetPoints(1), 20, ui64(4));
                CheckPoint(sample.GetPoints(2), 30, ui64(8));
            } else {
                FAIL() << "unknown metric: " << sample.ShortDebugString();
            }
        } else if (label0.GetName() == "requests") {
            ASSERT_EQ(label0.GetValue(), "count");
            ASSERT_EQ(sample.GetMetricType(), NProto::COUNTER);
            ASSERT_EQ(sample.PointsSize(), 1u);
            CheckPoint(sample.GetPoints(0), registryTs, ui64(1));
        } else if (label0.GetName() == "users") {
            ASSERT_EQ(label0.GetValue(), "count");
            ASSERT_EQ(sample.GetMetricType(), NProto::GAUGE);
            ASSERT_EQ(sample.PointsSize(), 1u);
            CheckPoint(sample.GetPoints(0), registryTs, 10.0);
        } else if (label0.GetName() == "usersInt") {
            ASSERT_EQ(label0.GetValue(), "count");
            ASSERT_EQ(sample.GetMetricType(), NProto::IGAUGE);
            ASSERT_EQ(sample.PointsSize(), 1u);
            CheckPoint(sample.GetPoints(0), registryTs, i64(100));
        } else if (label0.GetName() == "sensor") {
            if (label0.GetValue() == "bytes_rx") {
                ASSERT_EQ(sample.GetMetricType(), NProto::RATE);
                ASSERT_EQ(sample.PointsSize(), 1u);
                CheckPoint(sample.GetPoints(0), registryTs, ui64(20));
            } else if (label0.GetValue() == "hist_rate_explicit") {
                ASSERT_EQ(sample.GetMetricType(), NProto::HIST_RATE);
            } else if (label0.GetValue() == "hist_counter_linear") {
                ASSERT_EQ(sample.GetMetricType(), NProto::HISTOGRAM);
            } else {
                FAIL() << "unexpected metric: " << sample.ShortDebugString();
            }
        } else if (label0.GetName() == "consumer") {
            if (label0.GetValue() == "hist_counter_explicit") {
                ASSERT_EQ(sample.GetMetricType(), NProto::HISTOGRAM);
                ASSERT_EQ(sample.PointsSize(), 1u);
                ExpectHistogram(sample.GetPoints(0).GetHistogram(), {0, 5, Max<double>()}, {10, 20, 50});
            } else if (label0.GetValue() == "hist_rate_linear") {
                ASSERT_EQ(sample.GetMetricType(), NProto::HIST_RATE);
                ASSERT_EQ(sample.PointsSize(), 1u);
                ExpectHistogram(sample.GetPoints(0).GetHistogram(), {25, 40, Max<double>()}, {2, 3, 0});
            }
        } else {
            FAIL() << "unknown metric: " << sample.ShortDebugString();
        }
    }
}

TEST(TPython2PullTest, PullThrowing) {
    const TString module = R"(
class MyPullModule:
    def __init__(self, logger, registry, **params):
        pass

    def pull(self, ts, consumer):
        raise Exception()
)";

    TAgentLabels commonLabels = { {"os", "macOS"}, {"kernel", "Darwin"} };
    TPyModule holder{module, commonLabels};

    NProto::TMultiSamplesList samples;
    auto encoder = EncoderProtobuf(&samples);
    auto now = TInstant::Now();

    ASSERT_THROW(holder.Module->Pull(now, encoder.Get()), NPython2::TRuntimeError);
}

TEST(TPython2PullTest, IncorrectConsumerHistArgs) {
    NProto::TMultiSamplesList samples;
    auto encoder = EncoderProtobuf(&samples);

    {
        const TString module = R"(
class MyPullModule:
    def __init__(self, logger, registry, **params):
        pass

    def pull(self, ts, consumer):
        consumer.histogram_counter(
            {'consumer': 'hist_counter_explicit'}, [0, 5],
            buckets=[10, 20, 50])
)";
        TPyModule holder{module, {}};
        ASSERT_THROW(holder.Module->Pull(TInstant::Now(), encoder.Get()), NPython2::TRuntimeError);
    }

    {
        const TString module = R"(
class MyPullModule:
    def __init__(self, logger, registry, **params):
        pass

    def pull(self, ts, consumer):
        consumer.histogram_counter(
            {'consumer': 'hist_counter_explicit'}, [0, 5],
            type='linear',
            buckets=[10, 20, 50])
)";
        TPyModule holder{module, {}};
        ASSERT_THROW(holder.Module->Pull(TInstant::Now(), encoder.Get()), NPython2::TRuntimeError);
    }

    {
        const TString module = R"(
class MyPullModule:
    def __init__(self, logger, registry, **params):
        pass

    def pull(self, ts, consumer):
        consumer.histogram_counter(
            {'consumer': 'hist_counter_explicit'},
            type='explicit',
            buckets=[10, 20, 50])
)";
        TPyModule holder{module, {}};
        ASSERT_THROW(holder.Module->Pull(TInstant::Now(), encoder.Get()), NPython2::TRuntimeError);
    }

    {
        const TString module = R"(
class MyPullModule:
    def __init__(self, logger, registry, **params):
        pass

    def pull(self, ts, consumer):
        consumer.histogram_counter(
            {'consumer': 'hist_counter_explicit'}, 0,
            None,
            [10, 20, 50])
)";
        TPyModule holder{module, {}};
        ASSERT_THROW(holder.Module->Pull(TInstant::Now(), encoder.Get()), NPython2::TRuntimeError);
    }
    {
        const TString module = R"(
class MyPullModule:
    def __init__(self, logger, registry, **params):
        pass

    def pull(self, ts, consumer):
        consumer.histogram_counter(
            {'consumer': 'hist_counter_explicit'}, 0,
            [1],
            [10, 20, 50])
)";
        TPyModule holder{module, {}};
        ASSERT_THROW(holder.Module->Pull(TInstant::Now(), encoder.Get()), NPython2::TRuntimeError);
    }

    {
        const TString module = R"(
class MyPullModule:
    def __init__(self, logger, registry, **params):
        pass

    def pull(self, ts, consumer):
        consumer.histogram_counter(
            {'consumer': 'hist_counter_explicit'}, 0,
            [1],
            [80, 20, 50])
)";
        TPyModule holder{module, {}};
        ASSERT_THROW(holder.Module->Pull(TInstant::Now(), encoder.Get()), NPython2::TRuntimeError);
    }
}
