#include <solomon/agent/modules/pull/system/os_impl.h>
#include <solomon/agent/modules/pull/system/protos/system_config.pb.h>

#include <library/cpp/monlib/metrics/metric_consumer.h>
#include <library/cpp/monlib/metrics/metric_registry.h>
#include <library/cpp/monlib/metrics/labels.h>

#include <util/generic/string.h>
#include <util/generic/vector.h>
#include <util/generic/hash.h>
#include <util/datetime/base.h>
#include <util/generic/ptr.h>

#include <sys/sysctl.h>
#include <sys/mount.h>
#include <sys/stat.h>
#include <net/if_dl.h>

#include <CoreFoundation/CoreFoundation.h>
#include <IOKit/IOKitLib.h>
#include <IOKit/storage/IOMedia.h>
#include <IOKit/storage/IOBlockStorageDriver.h>
#include <IOKit/IOBSD.h>

#include <mach/mach_host.h>
#include <mach/mach_port.h>
#include <mach/mach.h>
#include <libproc.h>

#include <array>
#include <cerrno>
#include <ctime>

using namespace NMonitoring;


namespace NSolomon {
namespace NAgent {
namespace {

void UpdateRate(TRate* metric, ui64 val) {
    metric->Add(val - metric->Get());
}

template <typename T>
T ReadSysctlByName(TStringBuf name) {
    T val;
    size_t len = sizeof(val);

    if (sysctlbyname(name.data(), static_cast<void*>(&val), &len, nullptr, 0) < 0) {
        ythrow TSystemError() << "Call to sysctlbyname failed: " << strerror(errno);
    }

    return val;
}

template <typename T, size_t N>
T ReadSysctl(const std::array<int, N>& oid) {
    T val;
    size_t len = sizeof val;

    if (sysctl(oid.data(), oid.size(), &val, &len, nullptr, 0) < 0) {
        ythrow TSystemError() << "Call to sysctl failed: " << strerror(errno);
    }

    return val;
}

template <typename T, size_t N>
TVector<T> ReadSysctlMany(const std::array<int, N>& oid) {
    TVector<T> result;
    size_t size {0};

    auto* data = const_cast<int*>(oid.data());
    if (sysctl(data, oid.size(), nullptr, &size, nullptr, 0) == -1) {
        ythrow TSystemError() << "Call to sysctl failed: " << strerror(errno);
    }

    result.resize(size / sizeof(T));

    if (sysctl(data, oid.size(), result.data(), &size, nullptr, 0) == -1) {
        ythrow TSystemError() << "Call to sysctl failed: " << strerror(errno);
    }

    return result;
}

template <typename T>
T ReadHostStat(int what) {
    T val;
    mach_msg_type_number_t count = sizeof(T) / sizeof(integer_t);
    kern_return_t ret;

    auto port = mach_host_self();
    if (ret = host_statistics64(port, what, (host_info_t)&val, &count) != KERN_SUCCESS) {
        ythrow TSystemError() << "Call to host_statistics failed: " << ret;
    }

    mach_port_deallocate(mach_task_self(), port);

    return val;
}

/// CF helpers

TString ReadShortCfString(CFStringRef cfStr) {
    if (cfStr == nullptr) {
        return {};
    }

    char buf[64] = {};
    CFStringGetCString(cfStr, buf, sizeof(buf), CFStringGetSystemEncoding());

    return buf;
}

template <typename T>
T ReadCfDictUnsafe(CFDictionaryRef dict, CFStringRef what) {
    auto val = static_cast<T>(CFDictionaryGetValue(dict, what));

    return val;
}

template <typename T>
T ReadCfDict(CFDictionaryRef dict, CFStringRef what) {
    auto val = ReadCfDictUnsafe<T>(dict, what);
    Y_ENSURE_EX(val != nullptr, yexception() << "unable to read " << ReadShortCfString(what));

    return val;
}

/// Stat reading classes
/// For the source of inspiration refer to https://opensource.apple.com/source/top/top-111.20.1/libtop.c.auto.html

class TStatReader {
public:
    TStatReader(TMetricRegistry& metrics, TString path)
        : Metrics_{metrics}
        , Path_{path}
    {
    }

    virtual ~TStatReader() = default;

    virtual void Pull() = 0;

    const TString& GetPath() const {
        return Path_;
    }

    TGauge* CreateGauge(const TString& path, TLabels labels = {}) {
        labels.Add("path", GetPath() + path);
        return Metrics_.Gauge(labels);
    }

    TRate* CreateRate(const TString& path, TLabels labels = {}) {
        labels.Add("path", GetPath() + path);
        return Metrics_.Rate(labels);
    }

protected:
    TMetricRegistry& Metrics_;
    TString Path_;
};

class TCpuReader : public TStatReader {
public:
    TCpuReader(TMetricRegistry& metrics)
        : TStatReader{metrics, "/Cpu"}
        , FreqMin_{CreateGauge("/PhysicalCpuFrequency/Min")}
        , FreqAvg_{CreateGauge("/PhysicalCpuFrequency/Avg")}
        , FreqMax_{CreateGauge("/PhysicalCpuFrequency/Max")}
    {
    }

    void Pull() override {
        FreqMin_->Set(ReadSysctlByName<ui64>("hw.cpufrequency_min"));
        FreqAvg_->Set(ReadSysctlByName<ui64>("hw.cpufrequency"));
        FreqMax_->Set(ReadSysctlByName<ui64>("hw.cpufrequency_max"));
    }

private:
    TGauge* FreqMin_{nullptr};
    TGauge* FreqAvg_{nullptr};
    TGauge* FreqMax_{nullptr};
};

class TFilesystemReader : public TStatReader {
private:
    struct TFsStat {
        TGauge* Used{nullptr};
        TGauge* Free{nullptr};
        TGauge* Total{nullptr};
    };

public:
    TFilesystemReader(TMetricRegistry& metrics)
        : TStatReader{metrics, "/Filesystem"}
    {
    }

    void Pull() override {
        struct statfs *fsList;

        auto isDevfs = [] (const char* type) {
            return TStringBuf("devfs") == type;
        };

        const auto size = getmntinfo(&fsList, MNT_NOWAIT);
        for (auto i = 0; i < size; ++i) {
            auto& f = fsList[i];
            if ((f.f_flags & MNT_LOCAL) != MNT_LOCAL || isDevfs(f.f_fstypename)) {
                continue;
            }

            const TString mountpoint = f.f_mntonname;
            auto& stat = GetFsStat(mountpoint);
            stat.Total->Set(f.f_bsize * f.f_blocks);
            stat.Free->Set(f.f_bfree * f.f_bsize);
            stat.Used->Set((f.f_blocks - f.f_bfree) * f.f_bsize);
        }

    }

    TFsStat& GetFsStat(const TString& mountpoint) {
        auto it = FsMap_.find(mountpoint);

        if (it == std::end(FsMap_)) {
            std::tie(it, std::ignore) = FsMap_.emplace(mountpoint, TFsStat{});

            auto& stat = it->second;

            stat.Used = CreateGauge("/UsedB", {{"mountpoint", mountpoint}});
            stat.Free = CreateGauge("/FreeB", {{"mountpoint", mountpoint}});
            stat.Total = CreateGauge("/SizeB", {{"mountpoint", mountpoint}});
        }

        return it->second;
    }

private:
    THashMap<TString, TFsStat> FsMap_;
};

class TSystemReader : public TStatReader {
public:
    TSystemReader(TMetricRegistry& metrics)
        : TStatReader{metrics, "/System"}
        , Uptime_{CreateRate("/UpTime")}
        , UptimeRaw_{CreateGauge("/UpTimeRaw")}
        , UserTime_{CreateRate("/UserTime")}
        , SystemTime_{CreateRate("/SystemTime")}
        , IdleTime_{CreateRate("/IdleTime")}
        , NiceTime_{CreateRate("/NiceTime")}
    {
    }

    void Pull() {
        auto bootTime = ReadSysctlByName<struct timeval>("kern.boottime");
        const auto uptime = std::difftime(time(nullptr), bootTime.tv_sec) * 1000;
        UptimeRaw_->Set(uptime);
        UpdateRate(Uptime_, uptime);

        auto loadInfo = ReadHostStat<host_cpu_load_info_data_t>(HOST_CPU_LOAD_INFO);
        auto extractSecs = [&loadInfo](int idx) -> ui64 {
            return loadInfo.cpu_ticks[idx] / CLK_TCK;
        };

        UpdateRate(UserTime_, extractSecs(CPU_STATE_USER));
        UpdateRate(SystemTime_, extractSecs(CPU_STATE_SYSTEM));
        UpdateRate(NiceTime_, extractSecs(CPU_STATE_NICE));
        UpdateRate(IdleTime_, extractSecs(CPU_STATE_IDLE));
    }

private:
    TRate* Uptime_{nullptr};
    TGauge* UptimeRaw_{nullptr};
    TRate* UserTime_{nullptr};
    TRate* SystemTime_{nullptr};
    TRate* IdleTime_{nullptr};
    TRate* NiceTime_{nullptr};
};

struct TProcStats {
    ui64 Total;
    ui64 Blocked;
    ui64 Running;
    ui64 ThreadsTotal;
};

TProcStats GetProcStats() {
    TProcStats result = {};

    static constexpr std::array<int, 3> OID {{ CTL_KERN, KERN_PROC, KERN_PROC_ALL }};
    TVector<struct kinfo_proc> buf = ReadSysctlMany<struct kinfo_proc>(OID);

    for (auto i = 0u; i < buf.size(); ++i) {
        // pid == 0 is for kernel tasks
        if (buf[i].kp_proc.p_pid == 0) {
            continue;
        }

        result.Total++;

        struct proc_taskinfo pinf;
        if (proc_pidinfo(buf[i].kp_proc.p_pid, PROC_PIDTASKINFO, 0, &pinf, sizeof(pinf)) < 0) {
            continue;
        }

        result.ThreadsTotal += pinf.pti_threadnum;

        if (pinf.pti_numrunning > 0) {
            result.Running++;
        } else {
            result.Blocked++;
        }
    }

    return result;
}

class TProcReader: public TStatReader {
public:
    TProcReader(TMetricRegistry& metrics)
        : TStatReader{metrics, "/Proc"}
        , La_{CreateGauge("/La")}
        , Procs_{CreateGauge("/Procs")}
        , ProcsRunning_{CreateGauge("/ProcsRunning")}
        , ProcsBlocked_{CreateGauge("/ProcsBlocked")}
        , Threads_{CreateGauge("/Threads")}
    {
    }

    void Pull() {
        double la[3];
        if (getloadavg(la, sizeof(la)) < 0) {
            ythrow TSystemError() << "Call getload to getloadavg failed: " << strerror(errno);
        }

        La_->Set(la[1]);

        const auto procStats = GetProcStats();
        Procs_->Set(procStats.Total);
        ProcsBlocked_->Set(procStats.Blocked);
        ProcsRunning_->Set(procStats.Running);
        Threads_->Set(procStats.ThreadsTotal);

    }

private:
    TGauge* La_{nullptr};
    TGauge* Procs_{nullptr};
    TGauge* ProcsRunning_{nullptr};
    TGauge* ProcsBlocked_{nullptr};
    TGauge* Threads_{nullptr};
};

class TMemoryReader : public TStatReader {
public:
    TMemoryReader(TMetricRegistry& metrics)
        : TStatReader{metrics, "/Memory"}
        , MemTotal_{CreateGauge("/MemTotal")}
        , MemFree_{CreateGauge("/MemFree")}
        , SwapTotal_{CreateGauge("/SwapTotal")}
        , SwapFree_{CreateGauge("/SwapFree")}
        , Active_{CreateGauge("/Active")}
        , Inactive_{CreateGauge("/Inactive")}
        , PageIns_{CreateGauge("/PageIns")}
        , PageOuts_{CreateGauge("/PageOuts")}
        , SwapIns_{CreateGauge("/SwapIns")}
        , SwapOuts_{CreateGauge("/SwapOuts")}
        , MajorPageFaults_{CreateGauge("/MajorPageFaults")}
        , Unevictable_{CreateGauge("/Unevictable")}
        , AnonPageCount_{CreateGauge("/AnonPages")}
    {
    }

    void Pull() {
        const auto memTotal = ReadSysctlByName<ui64>("hw.memsize");
        MemTotal_->Set(memTotal);

        const auto swapUsage = ReadSysctlByName<struct xsw_usage>("vm.swapusage");
        SwapTotal_->Set(swapUsage.xsu_total);
        SwapFree_->Set(swapUsage.xsu_avail);

        const auto pageSize = ReadSysctlByName<ui64>("hw.pagesize");
        const auto vmInfo = ReadHostStat<vm_statistics64_data_t>(HOST_VM_INFO64);
        MemFree_->Set(vmInfo.free_count * pageSize);
        Active_->Set(vmInfo.active_count * pageSize);
        Inactive_->Set(vmInfo.inactive_count * pageSize);
        PageIns_->Set(vmInfo.pageins);
        PageOuts_->Set(vmInfo.pageouts);
        SwapIns_->Set(vmInfo.swapins);
        SwapOuts_->Set(vmInfo.swapouts);
        MajorPageFaults_->Set(vmInfo.faults);
        Unevictable_->Set(vmInfo.wire_count * pageSize);
        AnonPageCount_->Set(vmInfo.internal_page_count);

    }

private:
    TGauge* MemTotal_{nullptr};
    TGauge* MemFree_{nullptr};
    TGauge* SwapTotal_{nullptr};
    TGauge* SwapFree_{nullptr};
    TGauge* Active_{nullptr};
    TGauge* Inactive_{nullptr};
    TGauge* PageIns_{nullptr};
    TGauge* PageOuts_{nullptr};
    TGauge* SwapIns_{nullptr};
    TGauge* SwapOuts_{nullptr};
    TGauge* MajorPageFaults_{nullptr};
    TGauge* Unevictable_{nullptr};
    TGauge* AnonPageCount_{nullptr};
};

/// For more info refer to https://opensource.apple.com/source/IOStorageFamily/IOStorageFamily-116/IOBlockStorageDriver.h
class TBlockDevIterator {
public:
    TBlockDevIterator() {
        const auto ret = IOServiceGetMatchingServices(kIOMasterPortDefault, IOServiceMatching("IOBlockStorageDriver"), &Drives_);

        if (ret != kIOReturnSuccess) {
            ythrow TSystemError() << "Error in IOServiceGetMatchingServices(): " << ret;
        }
    }

    ~TBlockDevIterator() {
        IOObjectRelease(Drives_);
    }

    bool Next() {
        Ctx_ = {};
        CFStringRef diskNameRef {nullptr};

        do {
            Ctx_.Disk = IOIteratorNext(Drives_);

            if (Ctx_.Disk == 0) {
                return false;
            }

            Y_ENSURE_EX(
                IORegistryEntryCreateCFProperties(Ctx_.Disk,
                    (CFMutableDictionaryRef *)&Ctx_.Properties,
                    kCFAllocatorDefault,
                    kNilOptions) == kIOReturnSuccess,
                TSystemError() << "Error in IORegistryEntryCreateCFProperties()");

            Y_ENSURE_EX(Ctx_.Properties != nullptr, TSystemError() << "props are empty");

            diskNameRef = static_cast<CFStringRef>(IORegistryEntrySearchCFProperty(Ctx_.Disk,
                kIOServicePlane,
                CFSTR(kIOBSDNameKey),
                kCFAllocatorDefault,
                kIORegistryIterateRecursively));

            Ctx_.Stats = ReadCfDictUnsafe<CFDictionaryRef>(Ctx_.Properties, CFSTR(kIOBlockStorageDriverStatisticsKey));
        } while (diskNameRef == nullptr || Ctx_.Stats == nullptr);

        Ctx_.Name = ReadShortCfString(diskNameRef);

        return true;
    }

    void Reset() {
        IOIteratorReset(Drives_);
    }

    const TString& Name() const {
        return Ctx_.Name;
    }

    i64 Read(CFStringRef what) const {
        CFNumberRef cfnum = ReadCfDict<CFNumberRef>(Ctx_.Stats, what);

        i64 val {0};
        CFNumberGetValue(cfnum, kCFNumberSInt64Type, &val);

        return val;
    }

private:
    io_iterator_t Drives_ {0};

    struct TCtx {
        ~TCtx() {
            IOObjectRelease(Disk);

            if (Properties) {
                CFRelease(Properties);
            }

            // we don't own Stats_ though
        }

        io_registry_entry_t Disk {0};
        CFDictionaryRef Properties {nullptr};
        CFDictionaryRef Stats {nullptr};
        TString Name;
    };

    TCtx Ctx_;
};

class TDiskIoReader : public TStatReader {
private:
    struct TDiskStat {
        TGauge* Reads{nullptr};
        TGauge* ReadBytes{nullptr};
        TRate* ReadWaitMillisec{nullptr};
        TGauge* Writes{nullptr};
        TGauge* WriteBytes{nullptr};
        TRate* WriteWaitMillisec{nullptr};
    };

public:
    TDiskIoReader(TMetricRegistry& metrics)
        : TStatReader{metrics, "/Io/Disks"}
    {
    }

    void Pull() {
        TBlockDevIterator disk;
        while (disk.Next()) {
            auto& stat = GetDiskStat(disk.Name());
            stat.Reads->Set(disk.Read(CFSTR(kIOBlockStorageDriverStatisticsReadsKey)));
            stat.ReadBytes->Set(disk.Read(CFSTR(kIOBlockStorageDriverStatisticsBytesReadKey)));
            stat.Writes->Set(disk.Read(CFSTR(kIOBlockStorageDriverStatisticsWritesKey)));
            stat.WriteBytes->Set(disk.Read(CFSTR(kIOBlockStorageDriverStatisticsBytesWrittenKey)));

            // these calls return time measured in nanoseconds
            UpdateRate(stat.ReadWaitMillisec, disk.Read(CFSTR(kIOBlockStorageDriverStatisticsTotalReadTimeKey)) / 1e6);
            UpdateRate(stat.WriteWaitMillisec, disk.Read(CFSTR(kIOBlockStorageDriverStatisticsTotalWriteTimeKey)) / 1e6);
        }
    }

    TDiskStat& GetDiskStat(const TString& name) {
        auto it = DiskMap_.find(name);

        if (it == std::end(DiskMap_)) {
            std::tie(it, std::ignore) = DiskMap_.emplace(name, TDiskStat{});

            auto& stat = it->second;

            stat.Reads = CreateGauge("/Reads", {{"disk", name}});
            stat.ReadBytes = CreateGauge("/ReadBytes", {{"disk", name}});
            stat.ReadWaitMillisec = CreateRate("/ReadWaitMillisec", {{"disk", name}});
            stat.Writes = CreateGauge("/Writes", {{"disk", name}});
            stat.WriteBytes = CreateGauge("/WriteBytes", {{"disk", name}});
            stat.WriteWaitMillisec = CreateRate("/WriteWaitMillisec", {{"disk", name}});
        }

        return it->second;
    }

private:
    THashMap<TString, TDiskStat> DiskMap_;
};

// For more info refer to https://opensource.apple.com/source/network_cmds/network_cmds-77/ifconfig.tproj/ifconfig.c
class TNetIfIterator {
public:
    struct TIfStat {
        TString Name;
        ui64 RxBytes{0};
        ui64 TxBytes{0};
        ui64 RxPackets{0};
        ui64 TxPackets{0};
        ui64 RxErrors{0};
        ui64 TxErrors{0};
        ui64 RxDrop{0};
    };

public:
    TNetIfIterator() {
        static constexpr std::array<int, 6> OID = {{CTL_NET, PF_ROUTE, 0, 0, NET_RT_IFLIST, 0}};
        Raw_ = ReadSysctlMany<char>(OID);
        End_ = Raw_.data() + Raw_.size();
        Next_ = Raw_.data();
    }

    bool Next() {
        Current_ = {};

        if (Next_ >= End_) {
            return false;
        }

        auto* ifm = (struct if_msghdr *)Next_;
        Next_ += ifm->ifm_msglen;

        Y_ENSURE(ifm->ifm_type == RTM_IFINFO);
        while (Next_ < End_) {
            auto* nextifm = (struct if_msghdr *)Next_;

            if (nextifm->ifm_type != RTM_NEWADDR) {
                break;
            }

            Next_ += nextifm->ifm_msglen;
        }

        auto* sdl = (struct sockaddr_dl *)(ifm + 1);

        Current_.Name = TString{sdl->sdl_data, sdl->sdl_nlen};
        Current_.RxPackets = ifm->ifm_data.ifi_ipackets;
        Current_.TxPackets = ifm->ifm_data.ifi_opackets;
        Current_.RxBytes = ifm->ifm_data.ifi_ibytes;
        Current_.TxBytes = ifm->ifm_data.ifi_obytes;
        Current_.RxErrors = ifm->ifm_data.ifi_ierrors;
        Current_.TxErrors = ifm->ifm_data.ifi_oerrors;
        Current_.RxDrop = ifm->ifm_data.ifi_iqdrops;

        return true;
    }

    const TIfStat& Stats() const {
        return Current_;
    }

private:
    TVector<char> Raw_;
    const char* End_{nullptr};
    const char* Next_{nullptr};

    TIfStat Current_;
};

class TNetworkReader : public TStatReader {
private:
    struct TIfStat {
        TRate* RxBytes{nullptr};
        TRate* RxPackets{nullptr};
        TRate* RxErrs{nullptr};
        TRate* RxDrop{nullptr};
        TRate* TxBytes{nullptr};
        TRate* TxPackets{nullptr};
        TRate* TxErrs{nullptr};
    };

public:
    TNetworkReader(TMetricRegistry& metrics)
        : TStatReader{metrics, "/Net/Ifs"}
    {
    }

    void Pull() {
        TNetIfIterator netIfs;
        while (netIfs.Next()) {
            const auto& current = netIfs.Stats();
            auto& stats = GetIfStat(current.Name);

            UpdateRate(stats.RxBytes, current.RxBytes);
            UpdateRate(stats.RxPackets, current.RxPackets);
            UpdateRate(stats.RxErrs, current.RxErrors);
            UpdateRate(stats.RxDrop, current.RxDrop);
            UpdateRate(stats.TxBytes, current.TxBytes);
            UpdateRate(stats.TxPackets, current.TxPackets);
            UpdateRate(stats.TxErrs, current.TxErrors);
        }
    }

private:
    TIfStat& GetIfStat(const TString& name) {
        auto it = IfMap_.find(name);

        if (it == std::end(IfMap_)) {
            std::tie(it, std::ignore) = IfMap_.emplace(name, TIfStat{});

            auto& stat = it->second;

            stat.RxBytes = CreateRate("/RxBytes", {{"intf", name}});
            stat.RxErrs = CreateRate("/RxErrs", {{"intf", name}});
            stat.RxPackets = CreateRate("/RxPackets", {{"intf", name}});
            stat.RxDrop = CreateRate("/RxDrop", {{"intf", name}});
            stat.TxBytes = CreateRate("/TxBytes", {{"intf", name}});
            stat.TxErrs = CreateRate("/TxErrs", {{"intf", name}});
            stat.TxPackets = CreateRate("/TxPackets", {{"intf", name}});
        }

        return it->second;
    }

private:
    THashMap<TString, TIfStat> IfMap_;
};

class TDarwinPuller : public ISystemPullerImpl {
public:
    TDarwinPuller(TSystemConfig config)
        : Config_{config}
    {
        // no advanced metrics here at all

        if (config.GetCpu() != TSystemConfig::NONE) {
            Readers_.emplace_back(new TCpuReader{Metrics_});
        }

        if (config.GetStorage() != TSystemConfig::NONE) {
            Readers_.emplace_back(new TFilesystemReader{Metrics_});
        }

        if (config.GetKernel() != TSystemConfig::NONE) {
            Readers_.emplace_back(new TSystemReader{Metrics_});
            Readers_.emplace_back(new TProcReader{Metrics_});
        }

        if (config.GetMemory() != TSystemConfig::NONE) {
            Readers_.emplace_back(new TMemoryReader{Metrics_});
        }

        if (config.GetIo() != TSystemConfig::NONE) {
            Readers_.emplace_back(new TDiskIoReader{Metrics_});
        }

        if (config.GetNetwork() != TSystemConfig::NONE) {
            Readers_.emplace_back(new TNetworkReader{Metrics_});
        }
    }

    void Pull(IMetricConsumer* consumer) override {
        for (auto& reader: Readers_) {
            reader->Pull();
        }

        Metrics_.Append(TInstant::Zero(), consumer);
    }

private:
    TMetricRegistry Metrics_;
    TVector<THolder<TStatReader>> Readers_;
    TSystemConfig Config_;
};

} // namespace


ISystemPullerImplPtr CreateSysStatPuller(const TSystemConfig& config) {
    return ::MakeHolder<TDarwinPuller>(config);
}


} // namespace NAgent
} // namespace NSolomon
