#pragma once

#include <library/cpp/monlib/metrics/metric_type.h>

#include <util/generic/strbuf.h>
#include <util/datetime/base.h>

#include <tuple>
#include <type_traits>

namespace NSolomon::NAgent {
    template <class T, class Func>
    struct TMetricDescription {
        using MetricDataType = T;
        using FunctorType = Func;

        TStringBuf MetricName_;
        NMonitoring::EMetricType MetricType_;
        TStringBuf PropertyName_;
        FunctorType Functor_;

        constexpr TMetricDescription(
                TStringBuf metricName,
                NMonitoring::EMetricType metricType,
                TStringBuf propertyName,
                FunctorType&& functor)
            : MetricName_(metricName)
            , MetricType_(metricType)
            , PropertyName_(propertyName)
            , Functor_(functor)
        {
        }
    };

    template <class T>
    constexpr auto CreateMetricDescription(
            TStringBuf metricName,
            NMonitoring::EMetricType metricType,
            TStringBuf propertyName)
    {
        auto functor = [](const T& a) { return a; };
        return TMetricDescription<T, decltype(functor)>(metricName, metricType, propertyName, std::move(functor));
    }

    template <class T, class Func>
    constexpr auto CreateMetricDescription(
            TStringBuf metricName,
            NMonitoring::EMetricType metricType,
            TStringBuf propertyName,
            Func functor)
    {
        static_assert(std::is_same<typename std::result_of<Func(T)>::type, T>::value,
                      "The expression must return a value of type T.");
        return TMetricDescription<T, decltype(functor)>(metricName, metricType, propertyName, std::move(functor));
    }

    template <class... Metrics>
    struct TMetricGroup {
        using TMetricContainerType_ = std::tuple<Metrics...>;

        TStringBuf AccountingProperty_;
        TMetricContainerType_ Metrics_;

        constexpr TMetricGroup(TStringBuf accountingProperty, TMetricContainerType_&& metrics)
            : AccountingProperty_(accountingProperty)
            , Metrics_(metrics)
        {
        }
    };

    template <class... Args>
    constexpr TMetricGroup<Args...> MakeMetricGroup(TStringBuf accountingProperty, Args&&... args) {
        return TMetricGroup<Args...>(accountingProperty, std::forward_as_tuple(args...));
    }

    template <class MetricGroup, class Func>
    constexpr void Iterate(const MetricGroup& metricGroup, Func&& functor) {
        std::apply([&](auto... args){(functor(std::forward<decltype(args)>(args)), ...);}, metricGroup.Metrics_);
    }
}
