#include <solomon/agent/modules/pull/systemd/systemd.h>
#include <solomon/agent/modules/pull/systemd/worker.h>
#include <solomon/agent/modules/pull/systemd/protos/systemd_pull_config.pb.h>
#include <solomon/agent/misc/logger.h>

#include <library/cpp/monlib/encode/protobuf/protobuf.h>
#include <library/cpp/testing/gtest/gtest.h>

#include <util/generic/strbuf.h>

#include <contrib/libs/sdbus-cpp/include/sdbus-c++/sdbus-c++.h>

constexpr TStringBuf SYSTEMD_INTERFACE = "mock.yandex";
constexpr TStringBuf SERVICE_INTERFACE = "mock.yandex.Service";
constexpr TStringBuf MANAGER_INTERFACE = "mock.yandex.Manager";
constexpr TStringBuf DEFAULT_INTERFACE_PATH = "/mock/yandex";
constexpr TStringBuf MOCK_SERVICES_PATH = "/mock/yandex/yandexService";
constexpr TStringBuf MOCK_SERVICE_NAME = "yandexService.service";

class TManagerProvider {
public:
    TManagerProvider(sdbus::IConnection& connection) {
        SystemdObject_ = sdbus::createObject(connection, DEFAULT_INTERFACE_PATH.data());
        SystemdObject_->registerMethod("ListUnits").onInterface(MANAGER_INTERFACE.data()).implementedAs([](){
            std::vector<NSolomon::NAgent::SDBusReply> result;
            result.emplace_back(MOCK_SERVICE_NAME.data(), "", "", "", "running", "", MOCK_SERVICES_PATH.data(), 0, "", "/");
            result.emplace_back("yandexService1.service", "", "", "", "stopped", "", "/", 0, "", "/");
            result.emplace_back("testing.service", "", "", "", "running", "", "/", 0, "", "/");
            return result;
        });
        SystemdObject_->finishRegistration();
    }

private:
    std::unique_ptr<sdbus::IObject> SystemdObject_;
};

class TServiceProvider {
public:
    TServiceProvider(sdbus::IConnection& connection) {
        SystemdObject_ = sdbus::createObject(connection, MOCK_SERVICES_PATH.data());

        //CPU
        SystemdObject_->registerProperty("CPUAccounting").onInterface(SERVICE_INTERFACE.data()).withGetter([this](){
            return CPUAccounting_;
        });
        SystemdObject_->registerProperty("CPUUsageNSec").onInterface(SERVICE_INTERFACE.data()).withGetter([this](){
            return CPUUsageNSec_;
        });

        //Memory
        SystemdObject_->registerProperty("MemoryAccounting").onInterface(SERVICE_INTERFACE.data()).withGetter([this](){
            return MemoryAccounting_;
        });

        //Network
        SystemdObject_->registerProperty("IPAccounting").onInterface(SERVICE_INTERFACE.data()).withGetter([this](){
            return IPAccounting_;
        });
        SystemdObject_->registerProperty("IPEgressPackets").onInterface(SERVICE_INTERFACE.data()).withGetter([this](){
            return IPEgressPackets_;
        });
        SystemdObject_->registerProperty("IPIngressBytes").onInterface(SERVICE_INTERFACE.data()).withGetter([this](){
            return IPIngressBytes_;
        });
        SystemdObject_->registerProperty("IPIngressPackets").onInterface(SERVICE_INTERFACE.data()).withGetter([this](){
            return IPIngressPackets_;
        });
        SystemdObject_->finishRegistration();
    }

    ui64 GetCPUUsageNSec() {
        return CPUUsageNSec_;
    }

    ui64 GetIPEgressPackets() {
        return IPEgressPackets_;
    }

    ui64 GetIPIngressBytes() {
        return IPIngressBytes_;
    }

    ui64 GetIPIngressPackets() {
        return IPIngressPackets_;
    }

private:
    std::unique_ptr<sdbus::IObject> SystemdObject_;

    bool CPUAccounting_ = true;
    ui64 CPUUsageNSec_ = 3500000;
    bool MemoryAccounting_ = false;
    bool IPAccounting_ = true;
    ui64 IPEgressPackets_ = 4;
    ui64 IPIngressBytes_ = 5;
    ui64 IPIngressPackets_ = 6;
};

TEST(TSystemdUnitTest, Worker) {
    auto providersConnection = sdbus::createSessionBusConnection(SYSTEMD_INTERFACE.data());

    TManagerProvider managerProvider(*providersConnection);
    TServiceProvider serviceProvider(*providersConnection);
    providersConnection->enterEventLoopAsync();

    NSolomon::NAgent::TSystemdConfig config;
    config.set_pattern("yandex.*");
    config.set_cpu(NSolomon::NAgent::TSystemdConfig::ON);
    config.set_memory(NSolomon::NAgent::TSystemdConfig::ON);
    config.set_network(NSolomon::NAgent::TSystemdConfig::ON);

    NSolomon::NAgent::TSystemdSettings settings{
            .SystemdInterface = SYSTEMD_INTERFACE,
            .ServiceInterface = SERVICE_INTERFACE,
            .ManagerInterface = MANAGER_INTERFACE,
            .DefaultInterfacePath = DEFAULT_INTERFACE_PATH};

    auto workerConnection = sdbus::createSessionBusConnection();
    workerConnection->enterEventLoopAsync();
    auto worker = NSolomon::NAgent::CreateWorker(config, settings, *workerConnection);

    NMonitoring::NProto::TSingleSamplesList samples;
    auto encoder = NMonitoring::EncoderProtobuf(&samples);
    worker->GetMetrics(encoder.Get());
    ASSERT_EQ(samples.SamplesSize(), 4u);

    for (const auto& sample: samples.GetSamples()) {
        TString currentMetric;
        ASSERT_EQ(sample.LabelsSize(), 2u);
        {
            auto label1 = sample.GetLabels(0);
            ASSERT_EQ(label1.GetName(), "sensor");
            currentMetric = label1.GetValue();

            auto label2 = sample.GetLabels(1);
            ASSERT_EQ(label2.GetName(), "unit");
            ASSERT_EQ(label2.GetValue(), MOCK_SERVICE_NAME);
        }
        if (currentMetric == "cpuUsageMs") {
            ASSERT_EQ(sample.GetMetricType(), NMonitoring::NProto::RATE);
            ASSERT_EQ(sample.GetUint64(), serviceProvider.GetCPUUsageNSec() / 1000000);
        } else if (currentMetric == "txPackets") {
            ASSERT_EQ(sample.GetMetricType(), NMonitoring::NProto::RATE);
            ASSERT_EQ(sample.GetUint64(), serviceProvider.GetIPEgressPackets());
        } else if (currentMetric == "rxBytes") {
            ASSERT_EQ(sample.GetMetricType(), NMonitoring::NProto::RATE);
            ASSERT_EQ(sample.GetUint64(), serviceProvider.GetIPIngressBytes());
        } else if (currentMetric == "rxPackets") {
            ASSERT_EQ(sample.GetMetricType(), NMonitoring::NProto::RATE);
            ASSERT_EQ(sample.GetUint64(), serviceProvider.GetIPIngressPackets());
        } else {
            FAIL() << "Unexpected metric.";
        }
    }

    providersConnection->leaveEventLoop();
    workerConnection->leaveEventLoop();
}

TEST(TSystemdUnitTest, PullModule) {
    NSolomon::NAgent::TAgentLabels commonLabels{
            {"env", "prod"},
            {"dc", "man"}};
    NSolomon::NAgent::TSystemdConfig config;
    config.set_pattern("");

    auto pullModule = NSolomon::NAgent::CreateSystemdPullModule(commonLabels, config);

    NMonitoring::NProto::TSingleSamplesList samples;
    auto encoder = NMonitoring::EncoderProtobuf(&samples);
    pullModule->Pull(TInstant::Now(), encoder.Get());

    ASSERT_EQ(samples.CommonLabelsSize(), 2u);
    {
        auto label1 = samples.GetCommonLabels(0);
        ASSERT_EQ(label1.GetName(), "env");
        ASSERT_EQ(label1.GetValue(), "prod");

        auto label2 = samples.GetCommonLabels(1);
        ASSERT_EQ(label2.GetName(), "dc");
        ASSERT_EQ(label2.GetValue(), "man");
    }

    ASSERT_EQ(samples.SamplesSize(), 1u);
    auto sample = samples.GetSamples(0);
    ASSERT_EQ(sample.LabelsSize(), 2u);
    {
        auto label1 = sample.GetLabels(0);
        ASSERT_EQ(label1.GetName(), "sensor");
        ASSERT_EQ(label1.GetValue(), "version");

        auto label2 = sample.GetLabels(1);
        ASSERT_EQ(label2.GetName(), "systemdVersion");
    }
    {
        ASSERT_EQ(sample.GetMetricType(), NMonitoring::NProto::GAUGE);
        ASSERT_EQ(sample.GetUint64(), 1u);
    }
}
