#include "metrics.h"
#include "worker.h"

#include <solomon/agent/modules/pull/systemd/protos/systemd_pull_config.pb.h>
#include <solomon/agent/misc/logger.h>

#include <library/cpp/monlib/metrics/metric_consumer.h>

#include <util/generic/vector.h>
#include <util/datetime/base.h>

#include <contrib/libs/re2/re2/re2.h>

#include <optional>
#include <limits>

namespace NSolomon::NAgent {
    constexpr auto CpuMetrics = MakeMetricGroup(
            TStringBuf("CPUAccounting"),
            CreateMetricDescription<ui64>(
                    TStringBuf("cpuUsageMs"),
                    NMonitoring::EMetricType::RATE,
                    TStringBuf("CPUUsageNSec"),
                    [](const ui64& a){return a  / 1000000;}));

    constexpr auto MemoryMetrics = MakeMetricGroup(
            TStringBuf("MemoryAccounting"),
            CreateMetricDescription<ui64>(
                    TStringBuf("memoryUsageBytes"),
                    NMonitoring::EMetricType::GAUGE,
                    TStringBuf("MemoryCurrent")));

    constexpr auto NetworkMetrics = MakeMetricGroup(
            TStringBuf("IPAccounting"),
            CreateMetricDescription<ui64>(
                    TStringBuf("txBytes"),
                    NMonitoring::EMetricType::RATE,
                    TStringBuf("IPEgressBytes")),
            CreateMetricDescription<ui64>(
                    TStringBuf("txPackets"),
                    NMonitoring::EMetricType::RATE,
                    TStringBuf("IPEgressPackets")),
            CreateMetricDescription<ui64>(
                    TStringBuf("rxBytes"),
                    NMonitoring::EMetricType::RATE,
                    TStringBuf("IPIngressBytes")),
            CreateMetricDescription<ui64>(
                    TStringBuf("rxPackets"),
                    NMonitoring::EMetricType::RATE,
                    TStringBuf("IPIngressPackets")));

    template <class T>
    constexpr void CallConsumer(NMonitoring::IMetricConsumer* consumer, T value) {
        if constexpr (std::is_same_v<T, double>) {
            consumer->OnDouble(TInstant::Zero(), value);
        } else if constexpr (std::is_same_v<T, ui64>) {
            consumer->OnUint64(TInstant::Zero(), value);
        } else if constexpr (std::is_same_v<T, i64>) {
            consumer->OnInt64(TInstant::Zero(), value);
        } else {
            static_assert(TDependentFalse<T>);
        }
    }

    class SDBusServiceProxy {
    public:
        SDBusServiceProxy(
                const SDBusReply& reply,
                const TSystemdSettings& settings,
                sdbus::IConnection* connection)
            : Name_(reply.get<0>())
            , Path_(reply.get<6>())
            , Running_(reply.get<4>() == "running")
            , SystemdSettings_(settings)
            , SystemdConnection_(connection)
        {
        }

        const TString& GetName() const {
            return Name_;
        }

        bool IsRunning() const {
            return Running_;
        }

        void GetMetrics(
                NMonitoring::IMetricConsumer* consumer,
                const TSystemdConfig& config)
        {
            try {
                if (!SystemdConnection_) {
                    SystemdProxy_.Reset(sdbus::createProxy(
                            SystemdSettings_.SystemdInterface.data(),
                            Path_.c_str()).release());
                } else {
                    SystemdProxy_.Reset(sdbus::createProxy(
                            *SystemdConnection_,
                            SystemdSettings_.SystemdInterface.data(),
                            Path_.c_str()).release());
                }
            } catch (const sdbus::Error& e) {
                SA_LOG(WARN) << "Unable to connect to service " << Name_ << ". Interface "
                    << SystemdSettings_.SystemdInterface.data() << " is unavailable. " << e.getName() << ' '
                    << e.getMessage();
                return;
            }

            if (config.GetCpu() == TSystemdConfig::ON && AccountingEnabled(CpuMetrics.AccountingProperty_)) {
                Iterate(CpuMetrics, [this, consumer](const auto& metricDescription) {
                    OnMetric(consumer, metricDescription);
                });
            }

            if (config.GetMemory() == TSystemdConfig::ON && AccountingEnabled(MemoryMetrics.AccountingProperty_)) {
                Iterate(MemoryMetrics, [this, consumer](const auto& metricDescription) {
                    OnMetric(consumer, metricDescription);
                });
            }

            if (config.GetNetwork() == TSystemdConfig::ON && AccountingEnabled(NetworkMetrics.AccountingProperty_)) {
                Iterate(NetworkMetrics, [this, consumer](const auto &metricDescription) {
                    OnMetric(consumer, metricDescription);
                });
            }
        }

    private:
        template <class T>
        void OnMetric(NMonitoring::IMetricConsumer* consumer, const T& metricDescription) {
            auto metric = GetMetricValue<typename T::MetricDataType>(metricDescription.PropertyName_);
            if (!metric) {
                return;
            }

            auto value = metric.value();
            if (std::numeric_limits<typename T::MetricDataType>::max() == value) {
                // service unavailable
                return;
            }

            auto result = metricDescription.Functor_(value);
            consumer->OnMetricBegin(metricDescription.MetricType_);
            {
                consumer->OnLabelsBegin();
                consumer->OnLabel("sensor", metricDescription.MetricName_);
                consumer->OnLabel("unit", Name_);
                consumer->OnLabelsEnd();
            }
            CallConsumer(consumer, result);
            consumer->OnMetricEnd();
        }

        template <class T>
        std::optional<T> GetMetricValue(TStringBuf propertyName) {
            try {
                auto property = SystemdProxy_->getProperty(propertyName.data()).onInterface(SystemdSettings_.ServiceInterface.data());
                return {property.get<T>()};
            } catch (const sdbus::Error& e) {
                SA_LOG(WARN) << "Interface " << SystemdSettings_.ServiceInterface.data() << " of service " << Name_
                    << " is unavailable. Cannot read property " << propertyName << ". Error: " << e.getName()
                    << ' ' << e.getMessage();
                return {};
            }
        }

        bool AccountingEnabled(TStringBuf accountingProperty) {
            try {
                auto accProp = SystemdProxy_->getProperty(accountingProperty.data()).onInterface(SystemdSettings_.ServiceInterface.data());
                if (!accProp.get<bool>()) {
                    SA_LOG(WARN) << accountingProperty << " is disabled for " << Name_;
                    return false;
                }
            } catch (const sdbus::Error& e) {
                SA_LOG(WARN) << "Interface " << SystemdSettings_.ServiceInterface.data() << " of service " << Name_
                    << " is unavailable. Cannot read property " << accountingProperty << ". Error: " << e.getName()
                    << ' ' << e.getMessage();
                return false;
            }
            return true;
        }

    private:
        const TString Name_;
        const TString Path_;
        const bool Running_;
        THolder<sdbus::IProxy> SystemdProxy_;
        TSystemdSettings SystemdSettings_;
        sdbus::IConnection* SystemdConnection_;
    };

    class TWorker: public IWorker {
    public:
        TWorker(
                const TSystemdConfig& config,
                const TSystemdSettings& settings,
                sdbus::IConnection* connection = nullptr)
            : SystemdSettings_(settings)
            , SystemdConnection_(connection)
            , Config_{config}
            , Regexp_(new re2::RE2(config.GetPattern()))
        {
            Y_ENSURE(Regexp_->ok(), "Unable to compile regex " << config.GetPattern() << ": " << Regexp_->error());
        }

        void GetMetrics(NMonitoring::IMetricConsumer* consumer) override {
            if (!CheckDBusConnection()) {
                return;
            }

            if (GetSystemdVersion()) {
                PassSystemdVersion(consumer);
            }

            if (GetUnitList()) {
                for (auto& service: ActiveServices_) {
                    service.GetMetrics(consumer, Config_);
                }
            }

            ActiveServices_.clear();
        }

    private:
        bool CheckDBusConnection() {
            if (!SystemdProxy_) {
                try {
                    if (!SystemdConnection_) {
                        SystemdProxy_.Reset(sdbus::createProxy(
                                SystemdSettings_.SystemdInterface.data(),
                                SystemdSettings_.DefaultInterfacePath.data()).release());
                    } else {
                        SystemdProxy_.Reset(sdbus::createProxy(
                                *SystemdConnection_,
                                SystemdSettings_.SystemdInterface.data(),
                                SystemdSettings_.DefaultInterfacePath.data()).release());

                    }
                    return true;
                } catch (const sdbus::Error& e) {
                    SA_LOG(WARN) << "Unable to connect to service " << SystemdSettings_.DefaultInterfacePath.data()
                        << ": " << e.getName() << ' ' << e.getMessage();
                    SystemdProxy_.Reset();
                    return false;
                }
            }
            return true;
        }

        bool GetSystemdVersion() {
            if (SystemdVersion_.empty()) {
                try {
                    auto versionPropery = SystemdProxy_->getProperty("Version").onInterface(
                            SystemdSettings_.ManagerInterface.data());
                    SystemdVersion_ = versionPropery.get<std::string>();
                    return true;
                } catch (const sdbus::Error &e) {
                    SA_LOG(WARN) << "Interface " << SystemdSettings_.ManagerInterface.data() << " of service "
                        << SystemdSettings_.SystemdInterface.data()
                        << " is unavailable. Cannot read property Version. Error:" << e.getName()
                        << ' ' << e.getMessage();
                    return false;
                }
            }
            return true;
        }

        bool GetUnitList() {
            TVector<SDBusReply> serviceList;
            try {
                SystemdProxy_->callMethod("ListUnits").onInterface(SystemdSettings_.ManagerInterface.data()).storeResultsTo(serviceList);
            } catch (const sdbus::Error& e) {
                SA_LOG(WARN) << "Interface " << SystemdSettings_.ManagerInterface.data() << " of service "
                    << SystemdSettings_.SystemdInterface.data()
                    << " is unavailable. Cannot call method ListUnits. Error: "
                    << e.getName() << ' ' << e.getMessage();
                return false;
            }

            for (auto& i: serviceList) {
                SDBusServiceProxy tmp(i, SystemdSettings_, SystemdConnection_);
                if (!RE2::FullMatch(tmp.GetName(), *Regexp_)) {
                    continue;
                }
                if (tmp.IsRunning()) {
                    ActiveServices_.push_back(std::move(tmp));
                }
            }
            return !ActiveServices_.empty();
        }

        void PassSystemdVersion(NMonitoring::IMetricConsumer* consumer) {
            consumer->OnMetricBegin(NMonitoring::EMetricType::GAUGE);
            {
                consumer->OnLabelsBegin();
                consumer->OnLabel("sensor", "version");
                consumer->OnLabel("systemdVersion", SystemdVersion_);
                consumer->OnLabelsEnd();
            }
            consumer->OnUint64(TInstant::Zero(), 1);
            consumer->OnMetricEnd();
        }

    private:
        THolder<sdbus::IProxy> SystemdProxy_;
        TSystemdSettings SystemdSettings_;
        sdbus::IConnection* SystemdConnection_;
        TVector<SDBusServiceProxy> ActiveServices_;
        const TSystemdConfig Config_;
        THolder<re2::RE2> Regexp_;
        TString SystemdVersion_;
    };

    THolder<IWorker> CreateWorker(const class TSystemdConfig& config) {
        return THolder(new TWorker(config, {}));
    }

    THolder<IWorker> CreateWorker(
            const TSystemdConfig& config,
            const TSystemdSettings& settings,
            sdbus::IConnection& connection)
    {
        return MakeHolder<TWorker>(config, settings, &connection);
    }
}
