#include "unistat.h"

#include <solomon/agent/modules/pull/unistat/protos/unistat_pull_config.pb.h>
#include <solomon/agent/misc/logger.h>

#include <library/cpp/http/client/client.h>
#include <library/cpp/http/misc/httpcodes.h>

#include <library/cpp/monlib/encode/unistat/unistat.h>
#include <library/cpp/monlib/encode/json/json.h>
#include <library/cpp/monlib/encode/spack/spack_v1.h>

#include <util/datetime/base.h>
#include <util/string/builder.h>
#include <library/cpp/string_utils/url/url.h>

using namespace NMonitoring;

namespace NSolomon {
namespace NAgent {
namespace {

class TUnistatPullModule: public IPullModule {
public:
    TUnistatPullModule(const TLabels& labels, const TUnistatPullConfig& conf)
        : Name_{"UnistatPull/" + conf.GetUrl()}
        , Labels_{labels}
    {
        const auto& url = conf.GetUrl();
        TStringBuf scheme, host;
        ui16 port;

        Y_ENSURE(TryGetSchemeHostAndPort(url, scheme, host, port), "Url " << url << " is invalid");

        NHttp::TFetchOptions options;
        options.UserAgent = "SolomonAgent";
        options.RetryCount = conf.GetRetryCount();
        options.RetryDelay = TDuration::MilliSeconds(conf.GetRetryIntervalMillis());
        options.Timeout = TDuration::MilliSeconds(conf.GetTimeoutMillis() == 0
            ? 500
            : conf.GetTimeoutMillis());

        Query_ = NHttp::TFetchQuery{url, options};
    }

private:
    TStringBuf Name() const override {
        return Name_;
    }

    int Pull(TInstant, NMonitoring::IMetricConsumer* consumer) override {
        auto response = Fetch(Query_);

        if (response->Code != HTTP_OK) {
            SA_LOG(ERROR) << "Request to " << Query_.GetUrl() << " failed with code " << response->Code;
            return 0;
        }

        consumer->OnLabelsBegin();
        for (auto&& label: Labels_) {
            consumer->OnLabel(TString{label.Name()}, TString{label.Value()});
        }

        consumer->OnLabelsEnd();

        DecodeUnistat(response->Data, consumer);

        return 0;
    }

private:
    NHttp::TFetchQuery Query_;
    const TString Name_;
    const TLabels Labels_;
};

} // namespace


IPullModulePtr CreateUnistatPullModule(const TLabels& labels, const TUnistatPullConfig& config) {
    return new TUnistatPullModule(labels, config);
}

} // namespace NAgent
} // namespace NSolomon
