package ru.yandex.solomon.configs;

import java.io.BufferedReader;
import java.io.IOException;
import java.net.InetAddress;
import java.net.UnknownHostException;
import java.nio.file.Files;
import java.nio.file.Paths;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.OptionalInt;

import com.google.protobuf.Descriptors.Descriptor;
import com.google.protobuf.Descriptors.FieldDescriptor;
import com.google.protobuf.Message;
import com.google.protobuf.TextFormat;
import org.junit.Assert;
import org.junit.Test;

import ru.yandex.solomon.config.protobuf.TLoggingConfig;
import ru.yandex.solomon.config.protobuf.TThreadPoolConfig;
import ru.yandex.solomon.config.protobuf.TThreadPoolType;
import ru.yandex.solomon.config.protobuf.TThreadPoolsConfig;
import ru.yandex.solomon.config.protobuf.http.HttpServerConfig;
import ru.yandex.solomon.config.protobuf.rpc.TGrpcClientConfig;
import ru.yandex.solomon.config.thread.ThreadPoolProvider;
import ru.yandex.solomon.util.SolomonEnv;
import ru.yandex.solomon.util.net.KnownDc;


/**
 * @author Sergey Polovko
 */
public abstract class ValidateConfigsBase<TConfig extends Message> {

    private static final Map<SolomonEnv, Integer> hostsCount = new HashMap<>();
    static {
        hostsCount.put(SolomonEnv.DEVELOPMENT, 1);
        hostsCount.put(SolomonEnv.TESTING, 16);
        hostsCount.put(SolomonEnv.PRESTABLE, 32);
        hostsCount.put(SolomonEnv.PRODUCTION, 32);
    }

    private final String configFile;
    private final TConfig config;
    private final Optional<SolomonEnv> parsedEnvType;
    private final Optional<KnownDc> knownDc;

    private final FieldDescriptor httpServerConfig;
    private final FieldDescriptor threadPoolsConfig;
    private final FieldDescriptor loggingConfig;

    public ValidateConfigsBase(String configFile, TConfig.Builder configBuilder) {
        this.configFile = configFile;
        this.parsedEnvType = ValidateUtils.parseEnvType(configFile);
        this.knownDc = ValidateUtils.parseDc(configFile);

        Descriptor descriptorForType = configBuilder.getDescriptorForType();
        this.httpServerConfig = getField(descriptorForType, "HttpServerConfig");
        this.threadPoolsConfig = getField(descriptorForType, "ThreadPoolsConfig");
        this.loggingConfig = getField(descriptorForType, "LoggingConfig");

        this.config = readConfigFromFile(configBuilder, configFile);
    }

    private static FieldDescriptor getField(Descriptor descriptorForType, String fieldName) {
        return Objects.requireNonNull(
            descriptorForType.findFieldByName(fieldName),
            "config does not have '" + fieldName + "' field");
    }

    static <T extends Message> T readConfigFromFile(T.Builder configBuilder, String configFile) {
        try {
            try (BufferedReader reader = Files.newBufferedReader(Paths.get(configFile))) {
                TextFormat.merge(reader, configBuilder);
            }
        } catch (IOException e) {
            throw new RuntimeException(e);
        }
        //noinspection unchecked
        return (T) configBuilder.build();
    }

    protected TConfig getConfig() {
        return config;
    }

    protected String getConfigFile() {
        return configFile;
    }

    protected Optional<SolomonEnv> getParsedEnvType() {
        return parsedEnvType;
    }

    protected Optional<KnownDc> getKnownDc() {
        return knownDc;
    }

    @SuppressWarnings("unchecked")
    private <T> T getConfigField(FieldDescriptor field) {
        return (T) config.getField(field);
    }

    @Test
    public void validateHttpServerConfig() throws UnknownHostException {
        HttpServerConfig config = getConfigField(httpServerConfig);
        Assert.assertFalse(config.getBind().isEmpty());
        InetAddress.getByName(config.getBind());
        Assert.assertTrue(config.getPort() > 0);
        Assert.assertTrue(config.getThreadsCount() > 0);
    }

    @Test
    public void validateLoggingConfig() {
        TLoggingConfig logging = getConfigField(loggingConfig);
        String service = configFile.substring(configFile.lastIndexOf('/') + 1, configFile.indexOf("."));
        ValidateUtils.validateLoggingConfig(logging, parsedEnvType, service);
    }

    @Test
    public void validateSystemThreadPools() {
        validateThreadPoolName(ThreadPoolProvider.SystemThreadPools.IO.getName());
        validateThreadPoolName(ThreadPoolProvider.SystemThreadPools.SCHEDULER.getName());
    }

    protected void validateThreadPoolName(String threadPoolName) {
        TThreadPoolsConfig threadPools = getConfigField(threadPoolsConfig);
        List<TThreadPoolConfig> threadPoolsList = threadPools.getThreadPoolsList();
        long count = threadPoolsList.stream().map(TThreadPoolConfig::getName).filter(e -> e.equals(threadPoolName)).count();
        Assert.assertEquals("thread pool " + threadPoolName + " not found", count, 1);
    }

    protected void validateThreadPoolType(String threadPoolName, TThreadPoolType type) {
        TThreadPoolsConfig threadPools = getConfigField(threadPoolsConfig);
        List<TThreadPoolConfig> threadPoolsList = threadPools.getThreadPoolsList();
        TThreadPoolConfig threadPoolConfig = threadPoolsList.stream()
            .filter(tp -> tp.getName().equalsIgnoreCase(threadPoolName))
            .findFirst()
            .get();
        Assert.assertEquals(
            "thread pool " + threadPoolName + " has different type",
            type,
            threadPoolConfig.getThreadPoolType());
    }

    protected void validateGrpcClientConfig(TGrpcClientConfig config, int port) {
        ValidateUtils.validateListAddressesOrLocalHost(
            config.getAddressesList(),
            getCurrentEnvHostsCount(),
            true, true,
            getKnownDc(),
            OptionalInt.of(port));
        validateThreadPoolName(config.getThreadPoolName());
    }

    protected int getCurrentEnvHostsCount() {
        return parsedEnvType.isPresent()
            ? hostsCount.get(parsedEnvType.get())
            : 1; // special case for cloud;
    }
}
