package ru.yandex.solomon.configs;

import org.junit.Assert;
import org.junit.Test;
import org.junit.runner.RunWith;
import org.junit.runners.Parameterized;
import org.junit.runners.Parameterized.Parameters;

import ru.yandex.solomon.config.DataSizeConverter;
import ru.yandex.solomon.config.protobuf.TKikimrClientConfig;
import ru.yandex.solomon.config.protobuf.TStockpileClientConfig;
import ru.yandex.solomon.config.protobuf.coremon.TCoremonConfig;
import ru.yandex.solomon.config.protobuf.coremon.TCoremonEngineConfig;
import ru.yandex.solomon.config.protobuf.rpc.TGrpcServerConfig;
import ru.yandex.solomon.config.protobuf.rpc.TRpcServerConfig;


/**
 * @author alexlovkov
 */
@RunWith(Parameterized.class)
public class ValidateCoremonConfigsTest extends ValidateConfigsBase<TCoremonConfig> {

    @Parameters(name = "{0}")
    public static String[] configs() {
        return new String[]{
            "dev/coremon.dev.conf",
            "testing/coremon.conf",
            "prestable/coremon.conf",
            "production/coremon.sas.conf",
            "production/coremon.vla.conf"
        };
    }

    public ValidateCoremonConfigsTest(String configFile) {
        super(configFile, TCoremonConfig.newBuilder());
    }

    @Test
    public void validateCoremonBalancerConfig() {
        var config = getConfig().getBalancerConfig();
        validateGrpcClientConfig(config.getGrpcConfig(), Ports.COREMON_GRPC_INTERNAL);
    }

    @Test
    public void validateKikimrClientConfig() {
        TKikimrClientConfig config = getConfig().getKikimrClientConfig();
        ValidateUtils.validateKikimrClientConfig(config, getKnownDc(), getParsedEnvType(), true);
    }

    @Test
    public void validateStockpileClientConfig() {
        TStockpileClientConfig config = getConfig().getStockpileClientConfig();
        validateGrpcClientConfig(config.getGrpcConfig(), Ports.STOCKPILE_GRPC_PUBLIC);
    }

    @Test
    public void validateGrpcServerConfig() {
        TGrpcServerConfig config = getConfig().getGrpcServerConfig();
        Assert.assertEquals(Ports.METABASE_GRPC_PUBLIC, config.getPort(0));
        validateThreadPoolName(config.getThreadPoolName());
    }

    @Test
    public void validateRpcServerConfig() {
        TRpcServerConfig config = getConfig().getRpcServerConfig();
        {
            // (1) internal server
            Assert.assertTrue(config.hasInternalGrpcServerConfig());
            TGrpcServerConfig serverConfig = config.getInternalGrpcServerConfig();
            Assert.assertEquals(Ports.COREMON_GRPC_INTERNAL, serverConfig.getPort(0));
            validateThreadPoolName(serverConfig.getThreadPoolName());
            Assert.assertTrue(DataSizeConverter.toBytes(serverConfig.getMaxMessageSize()) >= (100 << 20)); // 100 MiB
        }
        // TODO: test public gRPC server
    }

    @Test
    public void validateCoremonEngineConfig() {
        TCoremonEngineConfig config = getConfig().getCoremonEngineConfig();
        validateThreadPoolName(config.getParsingThreadPool());
        validateThreadPoolName(config.getMiscThreadPool());
    }
}
