package ru.yandex.solomon.configs;

import java.util.Arrays;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.OptionalInt;
import java.util.regex.Pattern;

import com.google.common.net.HostAndPort;
import org.junit.Assert;
import org.junit.Test;
import org.junit.runner.RunWith;
import org.junit.runners.Parameterized;
import org.junit.runners.Parameterized.Parameters;

import ru.yandex.misc.lang.StringUtils;
import ru.yandex.solomon.config.protobuf.TAbcClientConfig;
import ru.yandex.solomon.config.protobuf.TClusterConfig;
import ru.yandex.solomon.config.protobuf.TKikimrClientConfig;
import ru.yandex.solomon.config.protobuf.TStockpileClientConfig;
import ru.yandex.solomon.config.protobuf.dataproxy.client.DataProxyClientConfig;
import ru.yandex.solomon.config.protobuf.frontend.TAuthConfig;
import ru.yandex.solomon.config.protobuf.frontend.TGatewayConfig;
import ru.yandex.solomon.config.protobuf.metabase.client.TMetabaseClientConfig;
import ru.yandex.solomon.config.protobuf.rpc.TGrpcClientConfig;
import ru.yandex.solomon.config.protobuf.rpc.TRpcClientConfig;
import ru.yandex.solomon.util.SolomonEnv;


/**
 * @author alexlovkov
 */
@RunWith(Parameterized.class)
public class ValidateGatewayConfigsTest extends ValidateConfigsBase<TGatewayConfig> {

    private static final Map<SolomonEnv, Integer> crossDcHostsCount = new HashMap<>();
    static {
        crossDcHostsCount.put(SolomonEnv.DEVELOPMENT, 1);
        crossDcHostsCount.put(SolomonEnv.TESTING, 16);
        crossDcHostsCount.put(SolomonEnv.PRESTABLE, 1);
        crossDcHostsCount.put(SolomonEnv.PRODUCTION, 9);
    }

    private static final List<String> expectedConfigsWithAbcClient = Arrays.asList(
        "testing/gateway.conf",
        "prestable/gateway.conf",
        "production/gateway.conf"
    );

    @Parameters(name = "{0}")
    public static String[] configs() {
        return new String[] {
            "dev/gateway.dev.conf",
            "testing/gateway.conf",
            "prestable/gateway.conf",
            "production/gateway.conf",
        };
    }

    public ValidateGatewayConfigsTest(String configFile) {
        super(configFile, TGatewayConfig.newBuilder());
    }

    @Test
    public void validateClientId() {
        String clientId = getConfig().getClientId();
        Assert.assertFalse(clientId.isEmpty());
    }

    @Test
    public void validateClustersConfig() {
        Pattern conductorGroupRe = Pattern.compile("^conductor_group://[a-z0-9_]+$");

        for (TClusterConfig clusterConfig : getConfig().getClustersConfigList()) {
            Assert.assertTrue(!StringUtils.isEmpty(clusterConfig.getClusterId()));

            if (getParsedEnvType().isPresent() && getParsedEnvType().get() == SolomonEnv.DEVELOPMENT) {
                Assert.assertEquals(1, clusterConfig.getAddressesCount());
                Assert.assertEquals("localhost", clusterConfig.getAddresses(0));
            } else {
                for (String address : clusterConfig.getAddressesList()) {
                    if (conductorGroupRe.matcher(address).matches()) {
                        // ok
                        break;
                    }
                }
            }
        }
    }

    @Test
    public void validateAuthConfig() {
        TAuthConfig config = getConfig().getAuthConfig();
        boolean emptyConfig = true;

        if (config.hasAsUserConfig()) {
            emptyConfig = false;
            SolomonEnv envType = getParsedEnvType().get();
            Assert.assertEquals(envType, SolomonEnv.DEVELOPMENT);
        }

        if (config.hasOAuthConfig()) {
            emptyConfig = false;
            TAuthConfig.TOAuthConfig oauthConfig = config.getOAuthConfig();
            Assert.assertFalse(oauthConfig.getClientId().isEmpty());
            Assert.assertFalse(oauthConfig.getClientSecret().isEmpty());

            SolomonEnv envType = getParsedEnvType().get();
            if (envType == SolomonEnv.TESTING) {
                Assert.assertEquals("solomon-test.yandex-team.ru", oauthConfig.getHost());
            } else if (envType == SolomonEnv.PRESTABLE) {
                Assert.assertEquals("solomon-prestable.yandex-team.ru", oauthConfig.getHost());
            } else if (envType == SolomonEnv.PRODUCTION) {
                Assert.assertEquals("solomon.yandex-team.ru", oauthConfig.getHost());
            } else {
                Assert.fail("unknown envType: " + envType);
            }
        }

        if (config.hasIamConfig()) {
            emptyConfig = false;
            TAuthConfig.TIamConfig iamConfig = config.getIamConfig();
            HostAndPort address = HostAndPort.fromString(iamConfig.getAccessServiceAddresses());
            Assert.assertTrue(
                "as.cloud.yandex-team.ru".equals(address.getHost()) ||
                "as.private-api.cloud-preprod.yandex.net".equals(address.getHost()) ||
                "as.private-api.cloud.yandex.net".equals(address.getHost()));
            Assert.assertEquals(4286, address.getPort());
        }

        if (emptyConfig) {
            Assert.fail("AuthConfig is empty");
        }
    }

    @Test
    public void validateFetcherClientConfig() {
        TRpcClientConfig config = getConfig().getFetcherClientConfig();
        if (config.hasGrpcConfig()) {
            validateGrpcClientConfig(config.getGrpcConfig(), Ports.FETCHER_GRCP_INTERNAL);
        } else {
            Assert.fail("bad fetcher client configuration");
        }
    }

    @Test
    public void validateCoremonClientConfig() {
        TRpcClientConfig config = getConfig().getCoremonClientConfig();
        if (config.hasGrpcConfig()) {
            validateGrpcClientConfig(config.getGrpcConfig(), Ports.COREMON_GRPC_INTERNAL);
        } else {
            Assert.fail("bad coremon client configuration");
        }
    }

    @Test
    public void validateCrossDcKikimrClientConfig() {
        TKikimrClientConfig config = getConfig().getCrossDcKikimrClientConfig();
        ValidateUtils.validateKikimrClientConfig(config, Optional.empty(), getParsedEnvType(), true);
        if (config.hasGrpcConfig()) {
            validateThreadPoolName(config.getGrpcConfig().getThreadPoolName());
        } else if (config.hasConnectionConfig()) {
            validateThreadPoolName(config.getConnectionConfig().getThreadPoolName());
        }
    }

    @Test
    public void validateDataProxyClientConfig() {
        DataProxyClientConfig config = getConfig().getDataProxyClientConfig();

        ValidateUtils.validateListAddressesOrLocalHost(
                config.getGrpcConfig().getAddressesList(),
                crossDcHostsCount.get(getParsedEnvType().get()),
                true, false, Optional.empty(),
                OptionalInt.of(Ports.DATAPROXY_GRPC_PUBLIC));
        validateThreadPoolName(config.getGrpcConfig().getThreadPoolName());
    }

    @Test
    public void validateMetabaseClientConfig() {
        TMetabaseClientConfig config = getConfig().getMetabaseClientConfig();
        ValidateUtils.validateListAddressesOrLocalHost(
            config.getGrpcConfig().getAddressesList(),
            crossDcHostsCount.get(getParsedEnvType().get()),
            true, false, Optional.empty(),
            OptionalInt.empty());
        validateThreadPoolName(config.getGrpcConfig().getThreadPoolName());
    }

    @Test
    public void validateStockpileClientConfig() {
        TStockpileClientConfig config = getConfig().getStockpileClientConfig();
        if (config.getGrpcConfig() != TGrpcClientConfig.getDefaultInstance()) {
            ValidateUtils.validateListAddressesOrLocalHost(
                config.getGrpcConfig().getAddressesList(),
                crossDcHostsCount.get(getParsedEnvType().get()),
                false, false, Optional.empty(), OptionalInt.empty());
            validateThreadPoolName(config.getGrpcConfig().getThreadPoolName());
        }
    }

    @Test
    public void validateAbcClientConfig() {
        TGatewayConfig config = getConfig();
        if (config.hasAbcClientConfig()) {
            TAbcClientConfig abcConfig = config.getAbcClientConfig();
            Assert.assertEquals("https://abc-back.yandex-team.ru", abcConfig.getUrl());
            Assert.assertTrue(expectedConfigsWithAbcClient.contains(getConfigFile()));
        }
    }
}
