package ru.yandex.solomon.configs;

import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.OptionalInt;
import java.util.stream.Collectors;

import com.google.common.net.HostAndPort;
import com.google.protobuf.ProtocolStringList;
import org.junit.Assert;

import ru.yandex.misc.lang.StringUtils;
import ru.yandex.solomon.config.protobuf.ELogToType;
import ru.yandex.solomon.config.protobuf.IamKeyAuthConfig;
import ru.yandex.solomon.config.protobuf.TConnectionConfig;
import ru.yandex.solomon.config.protobuf.TKikimrClientConfig;
import ru.yandex.solomon.config.protobuf.TLoggingConfig;
import ru.yandex.solomon.config.protobuf.TThreadPoolConfig;
import ru.yandex.solomon.config.protobuf.TThreadPoolsConfig;
import ru.yandex.solomon.config.protobuf.TvmAuthConfig;
import ru.yandex.solomon.config.protobuf.rpc.TGrpcClientConfig;
import ru.yandex.solomon.util.SolomonEnv;
import ru.yandex.solomon.util.net.KnownDc;

import static org.hamcrest.Matchers.startsWith;
import static org.junit.Assert.assertThat;

/**
 * @author alexlovkov
 */
public class ValidateUtils {

    private static final String CONDUCTOR_PREFIX = "conductor_group://";
    private static final String SOLOMON_PREFIX = "solomon://";
    private static final int KIKIMR_MSGBUS_PORT = 2134;
    private static final int KIKIMR_GRPC_PORT = 2135;

    private static final Map<SolomonEnv, Integer> hostsCount = new HashMap<>();
    private static final Map<SolomonEnv, ELogToType> envTypeToLogType = new HashMap<>();

    static {
        hostsCount.put(SolomonEnv.DEVELOPMENT, 1);
        hostsCount.put(SolomonEnv.TESTING, 16);
        hostsCount.put(SolomonEnv.PRESTABLE, 32);
        hostsCount.put(SolomonEnv.PRODUCTION, 32);

        envTypeToLogType.put(SolomonEnv.DEVELOPMENT, ELogToType.STDERR);
        envTypeToLogType.put(SolomonEnv.TESTING, ELogToType.FILE);
        envTypeToLogType.put(SolomonEnv.PRESTABLE, ELogToType.FILE);
        envTypeToLogType.put(SolomonEnv.PRODUCTION, ELogToType.FILE);
    }

    private static final String[] wellKnownThreadPools = new String[]{"CpuLowPriority", "CpuHighPriority", "Io", "Scheduler"};

    public static Optional<SolomonEnv> parseEnvType(String configName) {
        if (configName.contains("dev")) {
            return Optional.of(SolomonEnv.DEVELOPMENT);
        } else if (configName.contains("testing")) {
            return Optional.of(SolomonEnv.TESTING);
        } else if (configName.contains("prestable")) {
            return Optional.of(SolomonEnv.PRESTABLE);
        } else if (configName.contains("production")) {
            return Optional.of(SolomonEnv.PRODUCTION);
        } else {
            return Optional.empty();
        }
    }

    public static Optional<KnownDc> parseDc(String configName) {
        if (configName.contains("man")) {
            return Optional.of(KnownDc.MAN);
        } else if (configName.contains("myt")) {
            return Optional.of(KnownDc.MYT);
        } else if (configName.contains("testing")) {
            return Optional.of(KnownDc.SAS);
        } else if (configName.contains("prestable")) {
            return Optional.of(KnownDc.MAN);
        } else {
            return Optional.empty();
        }
    }

    // TODO: rethink this method
    public static void validateListAddressesOrLocalHost(
        ProtocolStringList addressesList,
        int addressesCount,
        boolean validateDC,
        boolean validateNumberPostfix,
        Optional<KnownDc> knownDc,
        OptionalInt expectedPort)
    {
        for (var address : addressesList) {
            if (address.startsWith(CONDUCTOR_PREFIX) || address.startsWith(SOLOMON_PREFIX) || address.startsWith("localhost")) {
                return;
            }
        }

        Assert.assertEquals(addressesList.size(), addressesCount);
        for (int i = 0; i < addressesList.size(); i++) {
            int index = i % 32;
            String address = addressesList.get(i);
            validateAddress(address, validateDC, knownDc, expectedPort);
            if (validateNumberPostfix) {
                Assert.assertTrue(address.contains(String.valueOf(index)));
            }
        }
    }

    private static void validateAddress(
        String address,
        boolean validateDC,
        Optional<KnownDc> knownDc,
        OptionalInt expectedPort)
    {
        HostAndPort hostAndPort = HostAndPort.fromString(address);
        Assert.assertTrue(hostAndPort.hasPort());
        if (validateDC && knownDc.isPresent()) {
            Assert.assertTrue(knownDc.map(e -> address.contains(e.name().toLowerCase())).orElse(false));
        }
        if (expectedPort.isPresent()) {
            Assert.assertEquals(expectedPort.getAsInt(), hostAndPort.getPort());
        }
    }

    private static void validateGrpcClientConfig(
        TGrpcClientConfig config,
        Optional<KnownDc> knownDc,
        Optional<SolomonEnv> parsedEnvType)
    {
        int addressesCount = parsedEnvType.isPresent()
            ? hostsCount.get(parsedEnvType.get())
            : 1; // special case for cloud
        ValidateUtils.validateListAddressesOrLocalHost(
            config.getAddressesList(), addressesCount, true, true, knownDc,
            OptionalInt.of(KIKIMR_GRPC_PORT));
    }

    private static void validateYdbConnection(TConnectionConfig config) {
        Assert.assertFalse("endpoint is not set", config.getEndpoint().isBlank());
        Assert.assertFalse("database is not set", config.getDatabase().isBlank());

        if (config.hasTvmAuth()) {
            TvmAuthConfig tvmAuth = config.getTvmAuth();
            Assert.assertNotEquals("tvm client id is not set", 0, tvmAuth.getClientId());
        } else if (config.hasIamKeyAuth()) {
            IamKeyAuthConfig iamKeyAuth = config.getIamKeyAuth();
            Assert.assertFalse("iam account id is not set", iamKeyAuth.getAccountId().isEmpty());
            Assert.assertFalse("iam key id is not set", iamKeyAuth.getKeyId().isEmpty());
            Assert.assertFalse("iam key path is not set", iamKeyAuth.getKeyPath().isEmpty());
        }

        config.getThreadPoolName();
    }

    public static void validateKikimrClientConfig(
        TKikimrClientConfig config,
        Optional<KnownDc> knownDc,
        Optional<SolomonEnv> parsedEnvType,
        boolean validateRootSchema)
    {
        if (config.hasGrpcConfig()) {
            validateGrpcClientConfig(config.getGrpcConfig(), knownDc, parsedEnvType);
        } else if (config.hasConnectionConfig()) {
            validateYdbConnection(config.getConnectionConfig());
            assertThat("invalid schema root",
                    config.getSchemaRoot(),
                    startsWith(config.getConnectionConfig().getDatabase()));
        } else {
            Assert.fail("GrpcConfig or ConnectionConfig must be present");
        }

        if (validateRootSchema) {
            validateSchemaRoot(config.getSchemaRoot());
        }
    }

    public static void validateSchemaRoot(String schemaRoot) {
        Assert.assertFalse(StringUtils.isBlank(schemaRoot));
        Assert.assertTrue(schemaRoot.startsWith("/"));
        Assert.assertFalse(schemaRoot.endsWith("/"));
    }

    public static void validateThreadPoolsConfig(TThreadPoolsConfig config) {
        List<TThreadPoolConfig> threadPoolsList = config.getThreadPoolsList();
        List<String> namesList = threadPoolsList.stream().map(TThreadPoolConfig::getName).collect(Collectors.toList());
        Assert.assertEquals(threadPoolsList.size(), namesList.stream().distinct().count());

        for (String defaultPool : wellKnownThreadPools) {
            Assert.assertTrue(namesList.contains(defaultPool));
        }
        for (TThreadPoolConfig pool : threadPoolsList) {
            Assert.assertNotEquals(0, pool.getThreads());
        }
    }

    public static void validateLoggingConfig(
        TLoggingConfig config,
        Optional<SolomonEnv> parsedEnvType,
        String service)
    {
        ELogToType logTo = config.getLogTo();
        if (parsedEnvType.isPresent()) {
            Assert.assertEquals(envTypeToLogType.get(parsedEnvType.get()), logTo);
            if (parsedEnvType.get() != SolomonEnv.DEVELOPMENT) {
                String file = config.getLogFile();
                Assert.assertEquals("/logs/" + service + ".log", file);
            }
        }

        Assert.assertTrue(config.getLoggersList().size() > 0);
    }
}
