#!/bin/sh -eux

STUFF_DIR="/tmp/files"


configure_base() {
    passwd -l root
    passwd -l ubuntu

    # Disable periodic APT updates
    echo 'APT::Periodic::Enable "0";' >> /etc/apt/apt.conf.d/10periodic

    # Wait for APT if it's already running
    flock /var/lib/apt/daily_lock true

    # Disable journald rate limits to avoid loss of any logs
    cat >> /etc/systemd/journald.conf << EOF
RateLimitInterval=0
RateLimitBurst=0
EOF
}

configure_network() {
    cat >> /etc/hosts.allow << EOF
sshd : [2a02:6b8:c0e:500:0:f847::]/96
sshd : [2a02:6b8:c02:900:0:f847::]/96
sshd : [2a02:6b8:c03:500:0:f847::]/96
EOF
    echo "sshd : ALL" >> /etc/hosts.deny

    cat >> /etc/hosts << EOF
169.254.169.254 metadata.google.internal
EOF

    rm -vf /etc/dhcp/dhclient-enter-hooks.d/resolvconf
}

configure_utils() {
    SKM_CLI="/usr/bin/skm"
    curl -s https://storage.cloud-preprod.yandex.net/skm/linux/skm -o $SKM_CLI
    chmod 755 $SKM_CLI

    YC_STORAGE_URL="https://storage.yandexcloud.net/yandexcloud-yc/release"
    YC_CLI="/usr/bin/yc"
    curl -s "${YC_STORAGE_URL}/$(curl -s "${YC_STORAGE_URL}/stable")/linux/amd64/yc" -o $YC_CLI
    chmod 755 $YC_CLI
}

configure_services() {
    export DEBIAN_FRONTEND=noninteractive

    # purge
    for p in yandex-push-client yandex-cauth libpam-sss sssd libnss-sss auditd ntp ntpdate salt-common salt-minion; do
        apt-get purge -yqq $p || true
    done
    apt-get autoremove -yqq
    rm -vrf /etc/sudoers.d/yandex-cauth* /srv/salt /var/cache/salt

    systemctl mask docker.service || true
    systemctl mask containerd.service || true
    systemctl mask kubelet.service || true

    systemctl mask yc-platform-set-fqdn.service || true
    systemctl mask systemd-journald-audit.socket || true
    systemctl restart systemd-journald-audit.socket || true

    # install
    apt-get update -qq
    apt-get install -y -o Acquire::ForceIPv6=yes \
        postfix \
        bsd-mailx \
        jq \
        osquery-vanilla=4.8.0.1 \
        osquery-yandex-generic-config=1.1.1.59 \
        apparmor-ycloud-svc-solomon-prof=1.0.6 \
        apparmor-ycloud-sssd-prof=1.0.5

    apt-get install -y -o Acquire::ForceIPv6=yes \
        yandex-solomon-agent-bin=1:18.0 \
        yandex-unified-agent \
        yandex-solomon-sysmond
        # yandex-ydb-cli

    apt-get install -y -o Acquire::ForceIPv6=yes \
        yandex-openjdk15=15.0.1-tzdata2020e \
        yandex-openjdk16=16.0.1-tzdata2021a

    for d in $DEB_EXTRACT_LIST; do
        if echo $d | grep -vEq \.deb$ ; then
            apt-get download $d
            d=${d%%=*}*.deb
        fi
        dpkg -x --no-debsig $d /
    done
    if [ -n "$PACKAGES_LIST" ]; then
        apt-get install -y -o Acquire::ForceIPv6=yes $PACKAGES_LIST
    fi
    for d in $DEB_INSTALL_LIST; do
        if echo $d | grep -vEq \.deb$ ; then
            apt-get download $d
            d=${d%%=*}*.deb
        fi
        dpkg -i --no-debsig $d
    done
    echo "DEB_EXTRACT_LIST: $DEB_EXTRACT_LIST"  > /var/log/versions.log
    echo "PACKAGES_LIST: $PACKAGES_LIST"       >> /var/log/versions.log
    echo "DEB_INSTALL_LIST: $DEB_INSTALL_LIST" >> /var/log/versions.log
}

cleanup_system() {
    # Update grub to specify correct fs UUID in configs
    GRUB_DISABLE_OS_PROBER=true update-grub

    # Clear apt caches
    apt-get clean all
    rm -vrf /var/lib/apt/lists/*

    rm -vrf /var/lib/cloud
    rm -vf /var/lib/dhcp/*.leases
    d_files="/etc/dhcp/dhclient-exit-hooks.d/ntp
             /etc/dhcp/dhclient-exit-hooks.d/ntpdate
             /run/ntp.conf.dhcp
             /var/lib/ntp/ntp.conf.dhcp"
    for d_file in ${d_files}; do
        deactivate_file "${d_file}"
    done
    rm -vrf $STUFF_DIR
    sync
    sleep 10
}

# ========================================================================================================================

deactivate_file() {
    if [ -e "${1}" ]; then
        truncate -s 0 "${1}" || true
        chattr +i "${1}" || true
    fi
}

# ========================================================================================================================

configure_base
configure_network
configure_services
configure_utils
cleanup_system
