#!/usr/bin/env python3
# -*- coding: UTF-8 -*-

import json
import yaml
import sys
import subprocess
import argparse
import os

'''
echo '{"default_cert_id": "b2551m9q88271398fine", "monitoring_api_cert_id": "b25cmu2879cvc8sf294q", "monitoring_private_api_cert_id": "b25cmu2879cvc8sf294q"}' | \
./files/skm_generator.py -c skm_test.conf -k c42nc82uin2g2vebgmj5 -y ~/.config/yav/.token -e israel --iam iam.private-api.yandexcloud.co.il:14283 --kms dpl.kms.private-api.yandexcloud.co.il:8443 --cert dpl.ycm.private-api.yandexcloud.co.il:8443 --lock cpl.lockbox.private-api.yandexcloud.co.il:8443 --dont-cleanup

from init.yaml to orig,conf:
python3 -c 'import yaml, sys; f = open(sys.argv[1]); c = yaml.safe_load(f); print("\n".join(["{}\t\t{}:{}\t{}\t{}://{}/{}".format(s["path"], s["uid"], s["gid"], oct(int(s["mode"]))[2:], *([[k, v["secretId"] if "secretId" in v else v["certificate_id"], v["key"]] for k, v in s["source"].items()][0])) for s in c["secrets"]]))' skm_stockpile.conf
'''

temp_file_path = "/dev/shm/skm.yaml"


format_dict = {
    "certificate_manager": {
        "id": "certificate_id",
        "key": "key"
    },
    "yav": {
        "id": "secretId",
        "key": "key"
    }
}


class run:
    def __init__(self, command, env={"LANG": "C"}):
        self.pid = -1
        self.out = ""
        self.err = ""
        self.status = 1
        env.update({"PATH": os.environ["PATH"]})
        try:
            self.p = subprocess.Popen(
                ["/bin/sh", "-c", " ".join(command)],
                env=env,
                stdout=subprocess.PIPE,
                stderr=subprocess.PIPE,
                close_fds=True,
                preexec_fn=lambda: os.setpgid(os.getpid(), os.getpid())
            )
            self.pid = self.p.pid
            try:
                self.out = self.p.stdout.read().strip().decode("utf-8")
                self.err = self.p.stderr.read().strip().decode("utf-8")
            except KeyboardInterrupt:
                os.killpg(self.pid, 2)
            self.p.wait()
            self.status = self.p.returncode
        except OSError as e:
            print("Got exception running {}: {}".format(command, e))
            exit(1)


# FILEPATH UID:GID MODE SRC://ID/KEY
def read_config(filename, replace):
    config = []
    with open(filename, "r") as file:
        for line in file:
            comment = line.find("#")
            flds = (line if comment < 0 else line[:comment]).replace("$", "").split()
            if len(flds) != 4:
                continue
            permissions = flds[1].split(":")
            src = flds[3].split("://")
            src_data = src[1].split("/")
            config.append({
                "path": flds[0].format(**replace),
                "uid": int(permissions[0]),
                "gid": int(permissions[1]),
                "mode": int(flds[2], 8),
                "source": {
                    src[0]: {
                        format_dict[src[0]]["id"]: src_data[0].format(**replace),
                        format_dict[src[0]]["key"]: src_data[1].format(**replace)
                    }
                }
            })
    return config


def main():
    parser = argparse.ArgumentParser(description="""
        if only --init is specified, use it to output secrets yaml
        if only --config is specified, use it to output secrets yaml, init file is created temporarily
        if both specified, use it to output secrets yaml, init file is saved to --init
    """)
    parser.add_argument("-d", "--dont-cleanup", action="store_true", help="do not cleanup temp file")
    parser.add_argument("-c", "--config", type=str, default="", metavar="FILE", help="config file path")
    parser.add_argument("-o", "--only-update-init", action="store_true", help="do not initialize init file, only update it")
    parser.add_argument("-i", "--init", type=str, default="", metavar="FILE", help="secrets init file")
    parser.add_argument("-k", "--kms-key-id", type=str, required=True, metavar="ID", help="kms key id")
    parser.add_argument("-y", "--yav-token-path", type=str, required=True, metavar="FILE", help="path to yav token file")
    parser.add_argument("-e", "--environment", type=str, required=True, metavar="ENV", help="ycp environment")
    parser.add_argument("--iam", type=str, required=True, metavar="IAM", help="iam private endpoint")
    parser.add_argument("--kms", type=str, required=True, metavar="KMS", help="kms private endpoint")
    parser.add_argument("--cert", type=str, required=True, metavar="CERT", help="cert manager private endpoint")
    parser.add_argument("--lock", type=str, required=True, metavar="LOCK", help="lockbox private endpoint")
    args = parser.parse_args()

    # check arguments
    if args.config == "" and args.init == "":
        print("Either --init or --config or both must be specified")
        return 1

    # input dict - variables for substitution
    try:
        input_dict = json.loads(sys.stdin.read())
    except Exception as e:
        print("Failed to parse input dict: {}".format(e))
        return 1

    try:
        with open(args.yav_token_path, "r") as file:
            yav_token = file.read()
    except Exception as e:
        print("Failed to read yav token file: {}".format(e))
        return 1

    r = run(["ycp", "--profile", args.environment, "iam", "create-token"])
    if r.status != 0:
        print("Failed to get ycp token ({}): {}".format(r.status, r.err))
        return 1
    yc_token = r.out

    # environment is ready
    init_file_path = temp_file_path if args.init == "" else args.init
    if args.config != "":
        try:
            secrets = read_config(args.config, input_dict)
        except Exception as e:
            print("Failed to read config: {}".format(e))
            return 1

        if not args.only_update_init:
            r = run([
                "skm", "init", init_file_path,
                "--kms-key-id", args.kms_key_id,
                "--iam-private-endpoint", args.iam,
                "--kms-private-endpoint", args.kms,
                "--certificate-manager-private-endpoint", args.cert,
                "--lockbox-private-endpoint", args.lock
            ], env={
                "YC_TOKEN": yc_token
            })
            if r.status != 0:
                print("Failed to init skm ({}): {}".format(r.status, r.err))
                return 1

        try:
            with open(init_file_path, "r") as file:
                config = yaml.safe_load(file.read())
        except Exception as e:
            print("Failed to read initialized config: {}".format(e))
            return 1

        config["secrets"] = secrets
        config["yav_token"] = {"env_var": "YAV_TOKEN"}
        with open(init_file_path, "w") as file:
            yaml.safe_dump(config, stream=file)

    r = run([
        "skm", "encrypt-md",
        "--config", init_file_path
    ], env={
        "YC_TOKEN": yc_token,
        "YAV_TOKEN":  yav_token
    })
    if r.status != 0:
        print("Failed to encrypt ({}): {}".format(r.status, r.err))
        return 1

    config = yaml.safe_load(yaml.safe_load(r.out)["skm"])
    print(yaml.dump(config, default_flow_style=False))

    if not args.dont_cleanup and args.init == "":
        try:
            os.unlink(temp_file_path)
        except Exception:
            pass


if __name__ == '__main__':
    exit(main())
