resource "time_sleep" "wait-sa" {
    lifecycle {
        prevent_destroy = true
    }
    create_duration = "10s"

    depends_on = [
        ycp_resource_manager_folder_iam_member.ig-sa-vpc,
        ycp_resource_manager_folder_iam_member.ig-sa-computeadmin,
        ycp_resource_manager_folder_iam_member.ig-sa-infra-computeadmin,
        ycp_resource_manager_folder_iam_member.ig-sa-gateway-lb,
        ycp_resource_manager_folder_iam_member.ig-sa-pm-lb,
        ycp_resource_manager_folder_iam_member.ig-sa-images-user,
        ycp_resource_manager_folder_iam_member.ig-sa-dns-editor,
        ycp_resource_manager_folder_iam_member.vm-sa-kms,
        ycp_resource_manager_folder_iam_member.vm-sa-lockbox,
        ycp_resource_manager_folder_iam_member.vm-sa-certificates
    ]
}

// ----------------------------------------------------------------------------

resource "ycp_iam_service_account" "images-sa" {
    lifecycle {
        prevent_destroy = true
    }
    folder_id = var.infra_folder_id
    name = "images-sa"
    service_account_id = "yc.monitoring.images-sa"
}

resource "ycp_resource_manager_folder_iam_member" "images-sa-images-user" {
    lifecycle {
        prevent_destroy = true
    }
    folder_id = var.infra_folder_id
    role = "compute.images.user"
    member = "serviceAccount:yc.monitoring.images-sa"
}

resource "ycp_resource_manager_folder_iam_member" "images-sa-service-images-user" {
    lifecycle {
        prevent_destroy = true
    }
    for_each = var.groups
    folder_id = each.value.folder_id
    role = "compute.editor"
    member = "serviceAccount:yc.monitoring.images-sa"
}

// for the account to update images in ig
resource "ycp_resource_manager_folder_iam_member" "images-sa-images-updater" {
    lifecycle {
        prevent_destroy = false
    }
    for_each = var.groups
    folder_id = each.value.folder_id
    role = "compute.admin"
    member = "serviceAccount:yc.monitoring.images-sa"
}

// ----------------------------------------------------------------------------

resource "ycp_iam_service_account" "ig-sa" {
    lifecycle {
        prevent_destroy = true
    }
    for_each = var.groups
    folder_id = each.value.folder_id
    name = "${each.key}-ig-sa"
    service_account_id = "yc.monitoring.${each.key}-ig-sa"
}

resource "ycp_resource_manager_folder_iam_member" "ig-sa-vpc" {
    lifecycle {
        prevent_destroy = true
    }
    for_each = ycp_iam_service_account.ig-sa
    folder_id = each.value.folder_id
    role = "vpc.viewer"
    member = "serviceAccount:${each.value.id}"
}

resource "ycp_resource_manager_folder_iam_member" "ig-sa-infra-computeadmin" {
    lifecycle {
        prevent_destroy = true
    }
    for_each = ycp_iam_service_account.ig-sa
    folder_id = var.infra_folder_id
    role = "internal.computeadmin"
    member = "serviceAccount:${each.value.id}"
}

resource "ycp_resource_manager_folder_iam_member" "ig-sa-computeadmin" {
    lifecycle {
        prevent_destroy = true
    }
    for_each = ycp_iam_service_account.ig-sa
    folder_id = each.value.folder_id
    role = "internal.computeadmin"
    member = "serviceAccount:${each.value.id}"
}

resource "ycp_resource_manager_folder_iam_member" "ig-sa-gateway-lb" {
    lifecycle {
        prevent_destroy = true
    }
    folder_id = var.groups["gateway"].folder_id
    role = "load-balancer.admin"
    member = "serviceAccount:yc.monitoring.gateway-ig-sa"
}

resource "ycp_resource_manager_folder_iam_member" "ig-sa-pm-lb" {
    lifecycle {
        prevent_destroy = true
    }
    folder_id = var.groups["pm"].folder_id
    role = "load-balancer.admin"
    member = "serviceAccount:yc.monitoring.pm-ig-sa"
}

resource "ycp_resource_manager_folder_iam_member" "ig-sa-iam-user" {
    lifecycle {
        prevent_destroy = true
    }
    for_each = ycp_iam_service_account.ig-sa
    folder_id = each.value.folder_id
    role = "iam.serviceAccounts.user"
    member = "serviceAccount:${each.value.id}"
}

resource "ycp_resource_manager_folder_iam_member" "ig-sa-images-user" {
    lifecycle {
        prevent_destroy = true
    }
    for_each = ycp_iam_service_account.ig-sa
    folder_id = each.value.folder_id
    role = "compute.images.user"
    member = "serviceAccount:${each.value.id}"
}

resource "ycp_resource_manager_folder_iam_member" "ig-sa-dns-editor" {
    lifecycle {
        prevent_destroy = true
    }
    for_each = ycp_iam_service_account.ig-sa
    folder_id = var.infra_folder_id
    role = "dns.editor"
    member = "serviceAccount:${each.value.id}"
}

// ----------------------------------------------------------------------------

locals {
    vmsas = {for group_name, group in var.groups : group_name => {
        //
        // use the same account, that was created at yc bootstrap stage
        // https://bb.yandex-team.ru/projects/CLOUD/repos/bootstrap-templates/browse/terraform/modules/monitoring/iam.tf
        //
        name = "${group_name}-sa"
        id = "yc.monitoring.${group_name}-sa"
        folder_id = group.folder_id
    }}
    // do not create: created at cloud bootstrap
    create_vm_sa = false
}

resource "ycp_iam_service_account" "vm-sa" {
    lifecycle {
        //prevent_destroy = true
    }
    for_each = local.create_vm_sa ? local.vmsas : {}
    folder_id = each.value.folder_id
    name = each.value.name
    service_account_id = each.value.id
}

resource "ycp_resource_manager_folder_iam_member" "vm-sa-kms" {
    lifecycle {
        prevent_destroy = true
    }
    for_each = local.vmsas
    folder_id = each.value.folder_id
    role = "kms.keys.encrypterDecrypter"
    member = "serviceAccount:${each.value.id}"
}

resource "ycp_resource_manager_folder_iam_member" "vm-sa-lockbox" {
    lifecycle {
        prevent_destroy = true
    }
    for_each = local.vmsas
    folder_id = each.value.folder_id
    role = "lockbox.payloadViewer"
    member = "serviceAccount:${each.value.id}"
}

resource "ycp_resource_manager_folder_iam_member" "vm-sa-certificates" {
    lifecycle {
        prevent_destroy = true
    }
    for_each = local.vmsas
    folder_id = each.value.folder_id
    role = "certificate-manager.certificates.downloader"
    member = "serviceAccount:${each.value.id}"
}

// ----------------------------------------------------------------------------

resource "ycp_iam_key" "sa-keys" {
    lifecycle {
        prevent_destroy = true
    }
    for_each = var.public-keys

    key_id = "yc.monitoring.${each.key}-key"
    service_account_id = "yc.monitoring.${each.key}"
    key_algorithm = "RSA_4096"
    format = "PEM_FILE"
    description = "Key for images account"
    public_key = each.value
}

// ----------------------------------------------------------------------------

resource "ycp_kms_symmetric_key" "symmetric-keys" {
    lifecycle {
        prevent_destroy = true
    }
    for_each = var.groups
    name = "${each.key}-kms-key"
    description = "key for secrets"
    folder_id = each.value.folder_id
    default_algorithm = "AES_256"
    status = "ACTIVE"
    labels = {}
}
