locals {
    subgroups = merge([
        for group_name, group in var.groups : {
            for idx, subgroup in group.subgroups : "${group_name}-${idx}" => {
                group_name = group_name
                user_data = "user-data.yaml"
                hostname = "${subgroup.name}.${var.dns_zone[var.env]}"
                shortname = subgroup.name
                zones = [ for dc, zone in subgroup.dc_zone : zone ]
                dc_map = jsonencode({ for dc, zone in subgroup.dc_zone : dc => var.ipv4_prefixes[zone] })
                disk_image_id = subgroup.disk_image_id
                ig_size = subgroup.ig_size
                folder_id = group.folder_id
                cores = group.cores
                memory = group.memory
                disk = group.disk
                logs = group.logs
                healthcheck_port = subgroup.healthcheck_port
                healthcheck_path = subgroup.healthcheck_path
            } if length(subgroup.disk_image_id) > 0 && subgroup.ig_size > 0
        }
    ]...)
    nsdomain = var.dns_zone[var.env]
}

resource "ycp_compute_placement_group" "monitoring-pg" {
    lifecycle {
        prevent_destroy = true
    }

    for_each = local.subgroups

    folder_id = each.value.folder_id
    name = "${each.key}-pg"
    spread_placement_strategy {
        best_effort = false
        max_instances_per_node = 1
        max_instances_per_fault_domain = 2
    }
}

resource "ycp_microcosm_instance_group_instance_group" "monitoring-ig" {
    lifecycle {
        prevent_destroy = true
    }

    for_each = local.subgroups

    name = "${each.key}-ig"
    service_account_id = "yc.monitoring.${each.value.group_name}-ig-sa"
    folder_id = each.value.folder_id

    deploy_policy {
        max_unavailable = 1
        max_expansion = 0
    }

    scale_policy {
        fixed_scale {
            size = each.value.ig_size
        }
    }

    allocation_policy {
        dynamic "zone" {
            for_each = toset(each.value.zones)
            content {
                zone_id = zone.value
            }
        }
    }

    depends_on = [
        time_sleep.wait-sa
    ]

    dynamic "load_balancer_spec" {
        for_each = each.value.group_name == "gateway" || each.value.group_name == "pm"  ? [0] : []
        content {
            target_group_spec {
                address_names = ["overlay-addr-v4", "overlay-addr-v6"]
                name = "${each.value.group_name}-target-group-${load_balancer_spec.value}"
            }
        }
    }

    dynamic "health_checks_spec" {
        for_each = each.value.healthcheck_port > 0 ? [0] : []
        content {
            
            dynamic "health_check_spec" {
                for_each = each.value.healthcheck_path == "" ? [0] : []
                content {
                    healthy_threshold = 10
                    unhealthy_threshold = 5
                    timeout = "2s"
                    interval = "5s"
                    tcp_options {
                        port = each.value.healthcheck_port
                    }
                }
            }
            dynamic "health_check_spec" {
                for_each = each.value.healthcheck_path != "" ? [0] : []
                content {
                    healthy_threshold = 10
                    unhealthy_threshold = 5
                    timeout = "2s"
                    interval = "5s"
                    http_options {
                        port = each.value.healthcheck_port
                        path = each.value.healthcheck_path
                        success_codes = [ 200 ]
                    }
                }
            }
            max_checking_health_duration = each.value.group_name == "gateway" ? "1200s" : "300s"
            // max_checking_health_duration = "300s"
        }
    }

    instance_template {
        name = each.value.shortname
        hostname = each.value.hostname
        fqdn = each.value.hostname

        platform_id = "standard-v3"
        //
        // use the same account, that was created at yc bootstrap stage
        // https://bb.yandex-team.ru/projects/CLOUD/repos/bootstrap-templates/browse/terraform/modules/monitoring/iam.tf
        //
        service_account_id = "yc.monitoring.${each.value.group_name}-sa"

        placement_policy {
            placement_group_id = ycp_compute_placement_group.monitoring-pg[each.key].id
        }

        resources {
            memory = each.value.memory
            cores  = each.value.cores
        }

        boot_disk {
            mode = "READ_WRITE"
            disk_spec {
                image_id = each.value.disk_image_id
                type_id = "network-hdd"
                size = each.value.disk
            }
        }

        secondary_disk {
            mode = "READ_WRITE"
            device_name = var.logs_disk_id
            disk_spec {
                description = "logs disk of ${each.value.hostname}"
                preserve_after_instance_delete = false
                type_id = "network-ssd"
                size = each.value.logs
            }
        }

        network_interface {
            network_id = ycp_vpc_network.monitoring-nets.id
            subnet_ids = values({ for name, subnet in ycp_vpc_subnet.monitoring-subnets : name => subnet.id })
            primary_v4_address {
                name = "overlay-addr-v4"
            }
            primary_v6_address {
                name = "overlay-addr-v6"
            }
        }

        underlay_network {
            network_name = "underlay-v6"
            ipv6_dns_record_spec {
                dns_zone_id = var.dns_zone_id
                fqdn = "${each.value.hostname}."
                ptr = var.set_ptr
                ttl = 600
            }
            dynamic "ipv6_dns_record_spec" {
                for_each = each.value.group_name == "gateway" ? [0] : []
                content {
                    dns_zone_id = var.dns_zone_id
                    fqdn = "${var.underlay_fqdn}."
                    ptr = false
                    ttl = 600
                }
            }
        }

        metadata = {
            serial-port-enable = 1
            skip_update_ssh_keys = true
            enable-oslogin = true
            shortname = each.value.shortname
            nsdomain = local.nsdomain
            k8s-runtime-bootstrap-yaml = ""
            ssh-keys = file("${path.module}/files/ssh-keys")
            user-data = templatefile("${path.module}/${each.value.user_data}", {
                service_env: var.service_env[var.env],
                service_dc_map: each.value.dc_map,
                dns_zone: var.dns_zone[var.env],
                hostname: each.value.hostname,
                disk_id: var.logs_disk_id,
            })
            skm = templatefile("${path.module}/skm_${each.value.group_name}.yaml", {
                default_cert_id: ycp_certificatemanager_certificate_request.yc-monitoring-gateway-cert["solomon"].id,
                pm_cert_id: ycp_certificatemanager_certificate_request.yc-monitoring-gateway-cert["pm"].id,
                monitoring_api_cert_id: ycp_certificatemanager_certificate_request.yc-monitoring-gateway-cert["monitoring-api"].id,
                monitoring_private_api_cert_id: ycp_certificatemanager_certificate_request.yc-monitoring-gateway-cert["monitoring-private-api"].id,
            })
        }
    }
}
