package main

import (
	"context"
	"encoding/base64"
	"fmt"
	"io"
	"os"
	"regexp"

	"a.yandex-team.ru/library/go/yandex/oauth"
	"a.yandex-team.ru/library/go/yandex/yav"
	"a.yandex-team.ru/library/go/yandex/yav/httpyav"
)

const (
	clientID     = "e397b846943844228aca41877a8b85d6"
	clientSecret = "aa15736d8bc8492c8bfa0578d9892f3f"
)

func readFile(filePath string) ([]byte, error) {
	file, err := os.Open(filePath)
	if err != nil {
		return []byte{}, err
	}
	defer file.Close()
	return io.ReadAll(file)
}

type Yav struct {
	ctx    context.Context
	verMap map[string]*yav.Version
	valMap map[string]string
	client *httpyav.Client
}

func NewYav() (*Yav, error) {
	y := &Yav{
		ctx:    context.Background(),
		verMap: make(map[string]*yav.Version),
		valMap: make(map[string]string),
	}
	token, err := oauth.GetTokenBySSH(y.ctx, clientID, clientSecret)
	if err != nil {
		return nil, fmt.Errorf("cannot get OAuth token by SSH, %v", err)
	}
	y.client, err = httpyav.NewClient(httpyav.WithOAuthToken(token))
	if err != nil {
		return nil, fmt.Errorf("cannot initialize yav client, %v", err)
	}
	return y, nil
}

func (y *Yav) Get(secID, secKey string) (*string, error) {
	secStr := fmt.Sprintf("%s/%s", secID, secKey)
	secVal, ok := y.valMap[secStr]
	if !ok {
		version, ok := y.verMap[secID]
		if !ok {
			v, err := y.client.GetVersion(y.ctx, secID)
			if err != nil {
				return nil, fmt.Errorf("cannot load secret %s, %v", secID, err)
			}
			version = &v.Version
			y.verMap[secID] = version
		}
		for _, v := range version.Values {
			if v.Key == secKey {
				secVal = v.Value
				y.valMap[secStr] = v.Value
			}
		}
		if len(secVal) == 0 {
			return nil, fmt.Errorf("secret %s does not have key %s", secID, secKey)
		}
	}
	return &secVal, nil
}

func gsubYav() (*string, error) {
	reSrc := regexp.MustCompile("\"?secrets_source://([^\"\\s]+)\"?")
	reYav := regexp.MustCompile("\"?yav://(sec-[a-z0-9]+)/([^\"\\s]+)\"?")

	y, err := NewYav()
	if err != nil {
		return nil, err
	}

	in, err := io.ReadAll(os.Stdin)
	if err != nil {
		return nil, fmt.Errorf("cannot read from stdin, %v", err)
	}
	inStr := string(in)

	srcMap := map[string]string{}
	for _, ss := range reSrc.FindAllStringSubmatch(inStr, -1) {
		srcStr := ss[0]
		srcPath := ss[1]

		if _, ok := srcMap[srcStr]; !ok {
			data, err := readFile(srcPath)
			if err != nil {
				return nil, err
			}
			secretsDataMap := map[string]string{}
			for _, ys := range reYav.FindAllStringSubmatch(string(data), -1) {
				secStr, secID, secKey := ys[0], ys[1], ys[2]
				secVal, err := y.Get(secID, secKey)
				if err != nil {
					return nil, err
				}
				secretsDataMap[secStr] = fmt.Sprintf("secrets: <\n  key: \"%s/%s\"\n  value: \"%s\"\n>\n", secID, secKey, *secVal)
			}
			secretsData := ""
			for _, s := range secretsDataMap {
				secretsData += s
			}
			srcMap[srcStr] = base64.StdEncoding.EncodeToString([]byte(secretsData))
		}
	}
	result := reSrc.ReplaceAllStringFunc(inStr, func(secret string) string {
		return srcMap[secret]
	})
	for _, ys := range reYav.FindAllStringSubmatch(result, -1) {
		secStr, secID, secKey := ys[0], ys[1], ys[2]
		secVal, err := y.Get(secID, secKey)
		if err != nil {
			return nil, err
		}
		srcMap[secStr] = base64.StdEncoding.EncodeToString([]byte(*secVal))
	}
	result = reYav.ReplaceAllStringFunc(result, func(secret string) string {
		return srcMap[secret]
	})
	return &result, nil
}

func main() {
	if result, err := gsubYav(); err != nil {
		_, _ = fmt.Fprint(os.Stderr, err)
		os.Exit(1)
	} else {
		_, _ = fmt.Printf("%s", *result)
	}
}
