import {Datasource} from "../module";
import Q from "q";

describe('GenericDatasource', function() {
    var ctx = {};

    beforeEach(function() {
        ctx.$q = Q;
        ctx.backendSrv = {};
        ctx.templateSrv = {};
        ctx.ds = new Datasource({}, ctx.$q, ctx.backendSrv, ctx.templateSrv);

    });

    it('should return an empty array when no targets are set', function(done) {
        ctx.ds.query({targets: []}).then(function(result) {
            expect(result.data).to.have.length(0);
            done();
        });
    });

    it('should return the server results when a target is set', function(done) {
        ctx.backendSrv.datasourceRequest = function(request) {
            return ctx.$q.when({
                _request: request,
                data: [
                    {
                        target: 'X',
                        datapoints: [1, 2, 3]
                    }
                ]
            });
        };

        ctx.templateSrv.replace = function(data) {
          return data;
        }

        ctx.ds.query({targets: ['hits']}).then(function(result) {
            expect(result._request.data.targets).to.have.length(1);

            var series = result.data[0];
            expect(series.target).to.equal('X');
            expect(series.datapoints).to.have.length(3);
            done();
        });
    });

    it ('should return empty metric results when a target is null', function(done) {
        ctx.backendSrv.datasourceRequest = function(request) {
            return ctx.$q.when({
                _request: request,
                data: [
                    "metric_0",
                    "metric_1",
                    "metric_2",
                ]
            });
        };

        ctx.templateSrv.replace = function(data) {
            return data;
        }

        ctx.ds.metricFindQuery({target: null}).then(function(result) {
            expect(result).to.have.length(0);
            done();
        });
    });

    it ('should return the metric target results when a target is set', function(done) {
        ctx.backendSrv.datasourceRequest = function(request) {
            var target = request.data.target;
            var result = 
            [{
                "cluster": "production",
                "delivery_service_id": "1003937",
                "delivery_type": "PICKUP",
                "project": "marketplace",
                "sensor": "/warehouse-capacities/by-delivery-service-and-warehouse",
                "service": "lms",
                "shipment_date": "DEFAULT",
                "warehouse_id": "47928"
            },
            {
                "cluster": "production",
                "delivery_service_id": "1003937",
                "delivery_type": "PICKUP",
                "project": "marketplace",
                "sensor": "/warehouse-capacities/by-delivery-service-and-warehouse",
                "service": "lms",
                "shipment_date": "DEFAULT",
                "warehouse_id": "218"
            },
            {
                "cluster": "production",
                "delivery_service_id": "1003937",
                "delivery_type": "PICKUP",
                "project": "marketplace",
                "sensor": "/warehouse-capacities/by-delivery-service-and-warehouse",
                "service": "lms",
                "shipment_date": "DEFAULT",
                "warehouse_id": "47806"
            }];

            return ctx.$q.when({
                _request: request,
                data: result
            });
        };

        ctx.templateSrv.replace = function(data) {
            return data;
        }

        ctx.ds.metricFindQuery({target: 'solomon_request#warehouse_id'}).then(function(result) {
            expect(result).to.have.length(3);
            expect(result[0].text).to.equal('47928');
            expect(result[0].value).to.equal('47928');
            expect(result[1].text).to.equal('218');
            expect(result[1].value).to.equal('218');
            expect(result[2].text).to.equal('47806');
            expect(result[2].value).to.equal('47806');
            done();
        });
    });

    it ('should return empty metric results when the target is an empty string', function(done) {
        ctx.backendSrv.datasourceRequest = function(request) {
            return ctx.$q.when({
                _request: request,
                data: [
                    "metric_0",
                    "metric_1",
                    "metric_2",
                ]
            });
        };

        ctx.templateSrv.replace = function(data) {
            return data;
        }

        ctx.ds.metricFindQuery({target: ''}).then(function(result) {
            expect(result).to.have.length(0);
            done();
        });
    });
    
    it ('should return empty metric results when the target is an empty string', function(done) {
        ctx.backendSrv.datasourceRequest = function(request) {
            return ctx.$q.when({
                _request: request,
                data: [
                    "metric_0",
                    "metric_1",
                    "metric_2",
                ]
            });
        };

        ctx.templateSrv.replace = function(data) {
            return data;
        }

        ctx.ds.metricFindQuery({target: ''}).then(function(result) {
            expect(result).to.have.length(0);
            done();
        });
    });

    it ('should return empty metric results when the args missing label name', function(done) {
        ctx.backendSrv.datasourceRequest = function(request) {
            return ctx.$q.when({
                _request: request,
                data: [
                    "metric_0",
                    "metric_1",
                    "metric_2",
                ]
            });
        };

        ctx.templateSrv.replace = function(data) {
            return data;
        }

        ctx.ds.metricFindQuery('solomon_request').then(function(result) {
            expect(result).to.have.length(0);
            done();
        });
    });
    
    it ('should return empty metric results when the label name is empty', function(done) {
        ctx.backendSrv.datasourceRequest = function(request) {
            return ctx.$q.when({
                _request: request,
                data: [
                    "metric_0",
                    "metric_1",
                    "metric_2",
                ]
            });
        };

        ctx.templateSrv.replace = function(data) {
            return data;
        }

        ctx.ds.metricFindQuery('solomon_request#').then(function(result) {
            expect(result).to.have.length(0);
            done();
        });
    });

    it ('should throw error when args are undefined', function(done) {
        global.assert.throw(ctx.ds.metricFindQuery, Error, "Cannot read property 'target' of undefined");
        done();
    });

    it ('should throw error when args are null', function(done) {
        global.assert.throw(function() { ctx.ds.metricFindQuery(null); }, Error, "Cannot read property 'target' of null");
        done();
    });

    it ('should return the metric target results when the args are a string', function(done) {
        ctx.backendSrv.datasourceRequest = function(request) {
            var target = request.data.target;
            var result = 
            [{
                "cluster": "production",
                "delivery_service_id": "1003937",
                "delivery_type": "PICKUP",
                "project": "marketplace",
                "sensor": "/warehouse-capacities/by-delivery-service-and-warehouse",
                "service": "lms",
                "shipment_date": "DEFAULT",
                "warehouse_id": "47928"
            },
            {
                "cluster": "production",
                "delivery_service_id": "1003938",
                "delivery_type": "PICKUP",
                "project": "marketplace",
                "sensor": "/warehouse-capacities/by-delivery-service-and-warehouse",
                "service": "lms",
                "shipment_date": "DEFAULT",
                "warehouse_id": "218"
            },
            {
                "cluster": "production",
                "delivery_service_id": "1003939",
                "delivery_type": "PICKUP",
                "project": "marketplace",
                "sensor": "/warehouse-capacities/by-delivery-service-and-warehouse",
                "service": "lms",
                "shipment_date": "DEFAULT",
                "warehouse_id": "47806"
            }];

            return ctx.$q.when({
                _request: request,
                data: result
            });
        };
        ctx.templateSrv.replace = function(data) {
            return data;
        }

        ctx.ds.metricFindQuery('solomon_request#delivery_service_id').then(function(result) {
            expect(result).to.have.length(3);
            expect(result[0].text).to.equal('1003937');
            expect(result[0].value).to.equal('1003937');
            expect(result[1].text).to.equal('1003938');
            expect(result[1].value).to.equal('1003938');
            expect(result[2].text).to.equal('1003939');
            expect(result[2].value).to.equal('1003939');
            done();
        });
    });
});
