import _ from "lodash";
import Selectors from "./utils/Selectors";

export class GenericDatasource {

    constructor(instanceSettings, $q, backendSrv, templateSrv) {
        this.type = instanceSettings.type;
        this.url = instanceSettings.url;
        this.name = instanceSettings.name;
        this.withCredentials = instanceSettings.withCredentials;
        this.q = $q;
        this.backendSrv = backendSrv;
        this.templateSrv = templateSrv;
        this.runResetSessionId();
    }

    runResetSessionId() {
        setInterval(() => this.resetSessionId(), 60000);
    }

    resetSessionId() {
        const options = {
            credentials: 'include',
        };

        return new Promise((resolve, reject) => {
            fetch('https://passport.yandex-team.ru/auth/update', options)
              .then((response) => {
                  if (response.status === 200) {
                      return response.json();
                  }

                  return reject();
              })
              .then((json) => {
                  if (json.status === 'ok') {
                      return resolve();
                  }

                  return reject();
              })
              .catch(() => reject());
        });
    }

    makeRequest(options) {
        if (this.withCredentials) {
            options.withCredentials = true;
        }
        return options;
    }

    query(options) {
        var query = this.buildQueryParameters(options);
        query.targets = query.targets.filter(t => !t.hide);

        if (query.targets.length <= 0) {
            return this.q.when({data: []});
        }

        var req = this.makeRequest({
            url: this.url + '/rest/grafana/search',
            data: query,
            method: 'POST',
            headers: {'Content-Type': 'application/json'}
        });

        return this.backendSrv.datasourceRequest(req);
    }

    testDatasource() {
        var req = this.makeRequest({
            url: this.url + '/',
            method: 'GET'
        });
        return this.backendSrv.datasourceRequest(req)
            .then(response => {
                if (response.status === 200) {
                    return {
                        status: "success",
                        message: "Data source is working",
                        title: "Success"
                    };
                }
            });
    }

    annotationQuery(options) {
        var query = this.templateSrv.replace(options.annotation.query, {}, 'glob');
        var annotationQuery = {
            range: options.range,
            annotation: {
                name: options.annotation.name,
                datasource: options.annotation.datasource,
                enable: options.annotation.enable,
                iconColor: options.annotation.iconColor,
                query: query
            },
            rangeRaw: options.rangeRaw
        };

        var req = this.makeRequest({
            url: this.url + '/annotations',
            method: 'POST',
            data: annotationQuery
        });
        return this.backendSrv.datasourceRequest(req)
            .then(result => {
                return result.data;
            });
    }

    metricFindQuery(options) {
        var target = typeof (options) === "string" ? options : options.target;
        if (!target) {
          return Promise.resolve([])
        }
        var reqParts = target.split("#")
        if (reqParts.length < 2) {
          return Promise.resolve([])
        }
        target = reqParts[0]
        var labelName = reqParts[1]
        if (!target || !labelName) {
          return Promise.resolve([])
        }

        var interpolated = this.templateSrv.replace(target, null, 'regex');
        var selectors = Selectors.parse(interpolated);
        var projectSelectors = selectors
            .getSelectors()
            .filter(selector => selector.getKey() === 'project');
        if (projectSelectors.length === 0) {
            return Promise.resolve([])
        }
        var project = projectSelectors[0].getValue();
        return this.metricsValueList(project, selectors, labelName);
    }

    metricsValueList(project, selectors, labelName) {
        var req = this.makeRequest({
            url: this.url + '/api/v2/projects/' + project + '/sensors/labels?limit=10000&names=' + labelName + '&selectors=' + encodeURIComponent(selectors.format(true)),
            method: 'GET',
            headers: {
                'accept': 'application/json',
                'Content-Type': 'application/json',
                'X-Yandex-Login': 'ya',
            }
        });

        return this.backendSrv.datasourceRequest(req)
            .then(result => {
                let labels = result.data.labels;
                if (labels.length !== 1) {
                    return {};
                }

                var res = _.map(labels[0].values, (d, i) => {
                    return {text: d, value: d}
                });

                if (labels[0]["absent"]) {
                    res.push({text: "-", value: "-"});
                }
                return res;
            });
    }

    sensorsKeyList(labellist) {
        var req = this.makeRequest({
            url: this.url + '/rest/grafana/sensors/key/list',
            data: labellist,
            method: 'POST',
            headers: {'Content-Type': 'application/json', 'X-Yandex-Login': 'ya'}
        });
        return this.backendSrv.datasourceRequest(req)
            .then(results => results.data);
    }

    sensorsValueList(labellist) {
        var req = this.makeRequest({
            url: this.url + '/rest/grafana/sensors/value/list',
            data: labellist,
            method: 'POST',
            headers: {'Content-Type': 'application/json', 'X-Yandex-Login': 'ya'}
        });
        return this.backendSrv.datasourceRequest(req)
            .then(results => results.data);
    }

    sensorsFindQuery(labellist) {
        var req = this.makeRequest({
            url: this.url + '/rest/grafana/sensors/list',
            data: labellist,
            method: 'POST',
            headers: {'Content-Type': 'application/json', 'X-Yandex-Login': 'ya'}
        });
        return this.backendSrv.datasourceRequest(req)
            .then(results => results.data);
    }

    buildQueryParameters(options) {
        let scopedVars = options.scopedVars;
        //remove placeholder targets
        options.targets = _.filter(options.targets, target => {
            return target.target !== 'select metric';
        });

        let targets = _.map(options.targets, target => {
            return {
                target: this.templateSrv.replace(target.target, scopedVars),
                refId: target.refId,
                hide: target.hide,
                type: target.type || 'timeserie'
            };
        });

        options.targets = targets;

        return options;
    }
}
