import { QueryCtrl } from 'app/plugins/sdk';
import kbn from 'app/core/utils/kbn';
import queryPart from './query_part.js';
import './css/query-editor.css!';
import { Trie } from './wildcard_trie';

export class GenericDatasourceQueryCtrl extends QueryCtrl {
    constructor($scope, $injector, uiSegmentSrv, $q, templateSrv) {
        super($scope, $injector);
        this.$q = $q;
        this.templateSrv = templateSrv;
        this.tagSegments = [];
        this.scope = $scope;
        this.uiSegmentSrv = uiSegmentSrv;
        this.target.target = this.target.target || 'select metric';
        this.target.type = this.target.type || 'timeserie';
        this.target.tags = this.target.tags || {};
        this.removeSegment = uiSegmentSrv.newSegment({fake: true, value: '-- remove --'});

        if (_.isEmpty(this.target.tags)) {
            this.tagSegments.push(uiSegmentSrv.newFake('select label', 'key', 'query-segment-key'));
            this.tagSegments.push(uiSegmentSrv.newOperator("="));
            this.tagSegments.push(uiSegmentSrv.newFake("select value", 'value', 'query-segment-value'));
        } else {
            let first = true;
            for (let key in this.target.tags) {
                if (this.target.tags.hasOwnProperty(key)) {
                    if (first) {
                        first = false;
                    } else {
                        this.tagSegments.push(this.uiSegmentSrv.newCondition('AND'));
                    }
                    this.tagSegments.push(uiSegmentSrv.newKey(key));
                    this.tagSegments.push(uiSegmentSrv.newOperator("="));
                    this.tagSegments.push(this.newLabelValue(this.target.tags[key]));
                }
            }
        }
        this.tagSegments.push(this.uiSegmentSrv.newPlusButton());
        this.target.groupBy = this.target.groupBy || [{type: 'time', params: ['$__interval']}];
        this.groupByParts = _.map(this.target.groupBy, queryPart.create);
        this.target.select = this.target.select || [[
                {type: 'field', params: ['value']},
                {type: 'avg', params: []},
            ]];
        this.selectModels = _.map(this.target.select, (parts) => {
            return _.map(parts, queryPart.create);
        });

        this.buildSelectMenu();
    }

    newLabelValue(value) {
        let result = this.uiSegmentSrv.newKeyValue(value);
        // hack without it asterics available values fail with exception $scope.getOptions(...) is undefined
        // https://github.com/grafana/grafana/issues/8277
        result.type = "value";
        return result;
    }

    buildSelectMenu() {
        let categories = queryPart.getCategories();
        this.selectMenu = _.reduce(categories, function (memo, cat, key) {
            let menu = {
                text: key,
                submenu: cat.map(item => {
                    return {text: item.type, value: item.type};
                }),
            };
            memo.push(menu);
            return memo;
        }, []);
    }

    addSelectPart(selectParts, cat, subitem) {
        let partModel = queryPart.create({type: subitem.value});
        partModel.def.addStrategy(selectParts, partModel, this);

        this.updatePersistedParts();
        this.rebuildQuery();
        this.refresh();
    }

    updatePersistedParts() {
        this.target.select = _.map(this.selectModels, function (selectParts) {
            return _.map(selectParts, function (part) {
                return {type: part.def.type, params: part.params};
            });
        });
    }


    //todo drop it?
    getOptions() {
        return this.datasource.sensorsFindQuery({})
            .then(this.uiSegmentSrv.transformToSegments(false));
    }

    getTagsOrValues(segment, index) {
        if (segment.type === 'operator') {
            return this.$q.when(this.uiSegmentSrv.newOperators(['=']));
        }

        let tagsCopy = _.mapValues(Object.assign({}, this.target.tags), (v) => {
            return this.templateSrv.replace(v, {});
        });

        // var query, addTemplateVars,
        // var transformFunction;
        if (segment.type === 'key' || segment.type === 'plus-button') {
            delete tagsCopy[segment.value];

            return this.datasource.sensorsKeyList(tagsCopy)
                .then(results => this.transformSensorKey(results))
                .catch(err => this.handleQueryError(err));

        } else if (segment.type === 'value') {
            let key = this.tagSegments[index - 2].value;
            tagsCopy[key] = "*";

            return this.datasource.sensorsValueList(tagsCopy)
                .then(results => this.extractSensorValue(index, results))
                .catch(err => this.handleQueryError(err));
        }
    }

    transformSensorKey(results) {
        if (results) {
            let r = [this.removeSegment];
            let segmentFactory = this.uiSegmentSrv;

            for (let key of results) {
                r.push(segmentFactory.newSegment({value: key, expandable: true}));
            }

            return r;
        } else {
            return [];
        }
    }

    extractSensorValue(index, results) {
        const trie = new Trie();

        let labelKey = this.tagSegments[index - 2].value;
        for (let labelList of results) {
            if (labelList.hasOwnProperty(labelKey)) {
                trie.add(labelList[labelKey]);
            }
        }

        let segmentFactory = this.uiSegmentSrv;
        return _.map(trie.getWords(), (key) => {
            return segmentFactory.newSegment({value: key, expandable: true});
        });
    }

    tagSegmentUpdated(segment, index) {
        this.tagSegments[index] = segment;

        let nextFocus;
        if (segment.type === 'plus-button') {
            nextFocus = index + 3;
        } else if (segment.type === 'key') {
            nextFocus = index + 2;
        } else if (segment.type === 'value') {
            nextFocus = index + 1;
        }


        // handle remove tag condition
        if (segment.value === this.removeSegment.value) {
            this.tagSegments.splice(index, 3);
            if (this.tagSegments.length === 0) {
                this.tagSegments.push(this.uiSegmentSrv.newPlusButton());
            } else if (this.tagSegments.length > 2) {
                this.tagSegments.splice(Math.max(index - 1, 0), 1);
                if (this.tagSegments[this.tagSegments.length - 1].type !== 'plus-button') {
                    this.tagSegments.push(this.uiSegmentSrv.newPlusButton());
                }
            }
        } else {
            if (segment.type === 'plus-button') {
                if (index > 2) {
                    this.tagSegments.splice(index, 0, this.uiSegmentSrv.newCondition('AND'));
                }
                this.tagSegments.push(this.uiSegmentSrv.newOperator('='));
                this.tagSegments.push(this.uiSegmentSrv.newFake('select tag value', 'value', 'query-segment-value'));
                segment.type = 'key';
                segment.cssClass = 'query-segment-key';
            }

            if ((index + 1) === this.tagSegments.length) {
                this.tagSegments.push(this.uiSegmentSrv.newPlusButton());
            }
        }

        this.setTagFocus(nextFocus);

        this.rebuildTargetTagConditions();
        this.panelCtrl.refresh();
    }

    setTagFocus(segmentIndex) {
        _.each(this.tagSegments, (segment, index) => {
            if (segmentIndex === index) {
                segment.focus = true;
            }
        });
    }

    buildSelector() {
        let selector = "{";
        let first = true;
        for (let key in this.target.tags) {
            if (this.target.tags.hasOwnProperty(key)) {
                let value = this.target.tags[key];

                if (first) {
                    first = false
                } else {
                    selector += ", ";
                }
                selector += "'" + key + "'";
                if (value.startsWith("!")) {
                    selector += "!=";
                    value = value.substr(1);
                } else {
                    selector += '=';
                }

                selector += "'" + value + "'";
            }
        }
        selector += "}";
        return selector;
    }

    rebuildQuery() {
        let selector = this.buildSelector();
        let groupTime = _.find(this.target.groupBy, (g) => g.type === 'time');
        let interval = "";
        if (groupTime) {
            interval = groupTime.params[0];
        }

        let parts = this.selectModels[0];
        let exprText = "";
        for (let index = 0; index < parts.length; index++) {
            let part = parts[index];
            exprText = part.def.renderer(part, exprText, selector, interval);
        }

        this.target.target = exprText;
    }

    rebuildTargetTagConditions() {
        let selectedLabels = {};

        _.each(this.tagSegments, (segment2, index) => {
            //todo skip fake key and value
            if (segment2.type === 'value' && !(segment2.hasOwnProperty('fake') && segment2.fake)) {
                selectedLabels[this.tagSegments[index - 2].value] = segment2.value;
            }
        });

        //todo normalize naming
        this.target.tags = selectedLabels;
        this.rebuildQuery();
    }

    gotoAdminSensors() {
        let selector = this.buildSelector();
        selector = this.templateSrv.replace(selector, this.getScopedVars());
        let project = this.target.tags["project"];
        return this.datasource.url + "/admin/projects/" + project + "/sensors/?selectors=" + encodeURIComponent(selector);
    }

    gotoAdminGraph() {
        let project = this.target.tags["project"];
        const begin = this.panelCtrl.range.from.utc().toISOString();
        const end = this.panelCtrl.range.to.utc().toISOString();
        let expression = this.templateSrv.replace(this.target.target, this.getScopedVars());
        return this.datasource.url + "/admin/projects/" + project + "/autoGraph?expression=" + encodeURIComponent(expression) + "&b=" + begin + "&e=" + end;
    }

    getScopedVars() {
        const res = kbn.calculateInterval(this.panelCtrl.range, 500, '1s');
        return Object.assign({}, this.panel.scopedVars, {
            __interval: { text: res.interval, value: res.interval },
            __interval_ms: { text: res.intervalMs, value: res.intervalMs },
        });
    }

    handleSelectPartEvent(selectParts, part, evt) {
        switch (evt.name) {
            case "part-param-changed": {
                this.rebuildQuery();
                this.refresh();
                break;
            }
            case "action": {
                this.removeSelectPart(selectParts, part);
                this.rebuildQuery();
                this.refresh();
                break;
            }
            case "get-part-actions": {
                return this.$q.when([{text: 'Remove', value: 'remove-part'}]);
            }
        }
    }

    removeSelectPart(selectParts, part) {
        if (part.def.type === 'field') {
            if (this.selectModels.length > 1) {
                var modelsIndex = _.indexOf(this.selectModels, selectParts);
                this.selectModels.splice(modelsIndex, 1);
            }
        } else {
            var partIndex = _.indexOf(selectParts, part);
            selectParts.splice(partIndex, 1);
        }

        this.updatePersistedParts();
        this.selectModels = _.map(this.target.select, (parts) => {
            return _.map(parts, queryPart.create);
        });
    }

    handleGroupByPartEvent(part, index, evt) {
        switch (evt.name) {
            case "part-param-changed": {
                this.target.groupBy[index] = part.part;
                this.rebuildQuery();
                this.refresh();
                break;
            }
            case "action": {
                this.removeGroupByPart(part, index);
                this.refresh();
                break;
            }
            case "get-part-actions": {
                return this.$q.when([{text: 'Remove', value: 'remove-part'}]);
            }
        }
    }

    removeGroupByPart(part, index) {
        var categories = queryPart.getCategories();

        if (part.def.type === 'time') {
            // remove fill
            this.target.groupBy = _.filter(this.target.groupBy, (g) => g.type !== 'fill');
            // remove aggregations
            this.target.select = _.map(this.target.select, (s) => {
                return _.filter(s, (part) => {
                    var partModel = queryPart.create(part);
                    if (partModel.def.category === categories.Aggregations) {
                        return false;
                    }
                    if (partModel.def.category === categories.Selectors) {
                        return false;
                    }
                    return true;
                });
            });
        }

        this.target.groupBy.splice(index, 1);
        this.groupByParts = _.map(this.target.groupBy, queryPart.create);
        this.rebuildQuery();
    }

    handleQueryError(err) {
        console.log(err);
        this.error = err.message || 'Failed to issue metric query';
        return [];
    }

    toggleEditorMode() {
        this.target.rawQuery = !this.target.rawQuery;
    }

    onChangeInternal() {
        this.panelCtrl.refresh(); // Asks the panel to refresh data.
    }
}

GenericDatasourceQueryCtrl.templateUrl = 'partials/query.editor.html';

