import _ from "lodash";

// import { functionRenderer } from 'app/core/components/query_part/query_part';
import {QueryPart, QueryPartDef, suffixRenderer} from "./query_part_utils";

var index = [];
var categories = {
    Aggregations: [],
    Combine: [],
    Rank: [],
    // Selectors: [],
    Transformations: [],
    Predictors: [],
    Math: [],
    Aliasing: [],
    Fields: [],
};

function createPart(part) {
    var def = index[part.type];
    if (!def) {
        throw {message: 'Could not find query part ' + part.type};
    }

    return new QueryPart(part, def);
}

function aliasRenderer(part, innerExpr) {
    return innerExpr + ' AS ' + '"' + part.params[0] + '"';
}

function register(options) {
    index[options.type] = new QueryPartDef(options);
    options.category.push(index[options.type]);
}

var groupByTimeFunctions = [];

function fieldRenderer(part, innerExpr, selector) {
    // if (part.params[0] === '*')  {
    //   return '*';
    // }
    // return '"' + part.params[0] + '"';
    return selector;
}

function aggrRenderer(part, innerExpr, selector, interval) {
    if (interval) {
        return `group_by_time(${interval}, '${part.def.type}', ${innerExpr})`;
    } else {
        return innerExpr;
    }
}

function groupLinesRenderer(part, innerExpr) {
    return `group_lines('${part.def.type}', ${innerExpr})`;
}

function replaceAggregationAddStrategy(selectParts, partModel) {
    // look for existing aggregation
    for (let i = 0; i < selectParts.length; i++) {
        let part = selectParts[i];
        if (part.def.category === categories.Aggregations) {
            selectParts[i] = partModel;
            return;
        }
    }

    selectParts.splice(1, 0, partModel);
}

function addCombineStrategy(selectParts, partModel) {
    // look for existing combine
    for (let i = 0; i < selectParts.length; i++) {
        let part = selectParts[i];
        if (part.def.category === categories.Combine) {
            selectParts[i] = partModel;
            return;
        }
    }

    let i;
    // look for index to add combine
    for (i = 0; i < selectParts.length; i++) {
        let part = selectParts[i];
        if (part.def.category === categories.Math || part.def.category === categories.Aliasing) {
            break;
        }
    }

    selectParts.splice(i, 0, partModel);
}

function addRankStrategy(selectParts, partModel) {
    // look for existing rank
    for (let i = 0; i < selectParts.length; i++) {
        let part = selectParts[i];
        if (part.def.category === categories.Rank) {
            selectParts[i] = partModel;
            return;
        }
    }

    let i;
    // look for index to add rank
    for (i = 0; i < selectParts.length; i++) {
        let part = selectParts[i];
        if (part.def.category === categories.Math || part.def.category === categories.Aliasing) {
            break;
        }
    }

    selectParts.splice(i, 0, partModel);
}

function addTransformationStrategy(selectParts, partModel) {
    var i;
    // look for index to add transformation
    for (i = 0; i < selectParts.length; i++) {
        var part = selectParts[i];
        if (part.def.category === categories.Math || part.def.category === categories.Aliasing) {
            break;
        }
    }

    selectParts.splice(i, 0, partModel);
}

function addMathStrategy(selectParts, partModel) {
    var partCount = selectParts.length;
    if (partCount > 0) {
        // if last is math, replace it
        if (selectParts[partCount - 1].def.type === 'math') {
            selectParts[partCount - 1] = partModel;
            return;
        }
        // if next to last is math, replace it
        if (selectParts[partCount - 2].def.type === 'math') {
            selectParts[partCount - 2] = partModel;
            return;
        } else if (selectParts[partCount - 1].def.type === 'alias') { // if last is alias add it before
            selectParts.splice(partCount - 1, 0, partModel);
            return;
        }
    }
    selectParts.push(partModel);
}

function addAliasStrategy(selectParts, partModel) {
    var partCount = selectParts.length;
    if (partCount > 0) {
        // if last is alias, replace it
        if (selectParts[partCount - 1].def.type === 'alias') {
            selectParts[partCount - 1] = partModel;
            return;
        }
    }
    selectParts.push(partModel);
}

function addFieldStrategy(selectParts, partModel, query) {
    // copy all parts
    var parts = _.map(selectParts, function (part) {
        return createPart({type: part.def.type, params: _.clone(part.params)});
    });

    query.selectModels.push(parts);
}

function functionRendererPreviousLast(part, innerExpr) {
    let str = part.def.type + '(';
    let parameters = prepareParams(part);
    if (innerExpr) {
        parameters.push(innerExpr);
    }
    return str + parameters.join(', ') + ')';
}

function functionRendererPreviousLastWithOptionalVector(part, innerExpr) {
    let str = part.def.type + '(';
    let parameters = prepareParams(part);
    if (innerExpr) {
        parameters.push(innerExpr);
    }

    if (parameters.length > 0 && parameters[0].indexOf(',') > 0) {
        parameters[0] = 'as_vector(' + parameters[0] + ')';
    }

    return str + parameters.join(', ') + ')';
}

function prepareParams(part) {
    return _.map(part.params, (value, index) => {
        let paramType = part.def.params[index];
        if (paramType.type === 'time') {
            if (value === 'auto') {
                value = '$__interval';
            }
        }
        if (paramType.quote === 'single') {
            return "'" + value + "'";
        } else if (paramType.quote === 'double') {
            return '"' + value + '"';
        }

        return value;
    });
}

function functionRenderer(part, innerExpr) {
    let str = part.def.type + '(';
    let parameters = _.map(part.params, (value, index) => {
        let paramType = part.def.params[index];
        if (paramType.type === 'time') {
            if (value === 'auto') {
                value = '$__interval';
            }
        }
        if (paramType.quote === 'single') {
            return "'" + value + "'";
        } else if (paramType.quote === 'double') {
            return '"' + value + '"';
        }

        return value;
    });

    if (innerExpr) {
        parameters.unshift(innerExpr);
    }
    return str + parameters.join(', ') + ')';
}

register({
    type: 'field',
    addStrategy: addFieldStrategy,
    category: categories.Fields,
    params: [{type: 'field', dynamicLookup: true}],
    defaultParams: ['value'],
    renderer: fieldRenderer,
});

// Aggregations
register({
    type: 'max',
    addStrategy: replaceAggregationAddStrategy,
    category: categories.Aggregations,
    params: [],
    defaultParams: [],
    renderer: aggrRenderer,
});

register({
    type: 'min',
    addStrategy: replaceAggregationAddStrategy,
    category: categories.Aggregations,
    params: [],
    defaultParams: [],
    renderer: aggrRenderer,
});

register({
    type: 'avg',
    addStrategy: replaceAggregationAddStrategy,
    category: categories.Aggregations,
    params: [],
    defaultParams: [],
    renderer: aggrRenderer,
});

register({
    type: 'sum',
    addStrategy: replaceAggregationAddStrategy,
    category: categories.Aggregations,
    params: [],
    defaultParams: [],
    renderer: aggrRenderer,
});

register({
    type: 'last',
    addStrategy: replaceAggregationAddStrategy,
    category: categories.Aggregations,
    params: [],
    defaultParams: [],
    renderer: aggrRenderer,
});

register({
    type: 'count',
    addStrategy: replaceAggregationAddStrategy,
    category: categories.Aggregations,
    params: [],
    defaultParams: [],
    renderer: aggrRenderer,
});

//Combine
register({
    type: 'group_lines',
    addStrategy: addCombineStrategy,
    category: categories.Combine,
    params: [{name: "aggregator", quote: 'single', type: "string", options: ['max', 'min', 'avg', 'sum', 'last', 'std', 'count', 'integrate', 'random']}],
    defaultParams: ['sum'],
    renderer: functionRendererPreviousLast,
});

register({
    type: 'group_by_labels',
    addStrategy: addCombineStrategy,
    category: categories.Combine,
    params: [
        {name: "aggregator", type: "string", options: ['max', 'min', 'avg', 'sum', 'last', 'std', 'count', 'integrate', 'random']},
        {name: "labels", type: "string", quote: 'double'},
    ],
    defaultParams: ['sum', 'host'],
    renderer: (part, innerExpr) => {
        let parameters = prepareParams(part);
        if (innerExpr) {
            parameters.push(innerExpr);
        }

        return parameters[0] + "(" + innerExpr + ") by (" + parameters[1] + ")";
    },
});

register({
    type: 'percentile_group_lines',
    addStrategy: addCombineStrategy,
    category: categories.Combine,
    params: [{name: "percentile", type: 'double', options: ['50', '80', '99', '99.9']}],
    defaultParams: ['90'],
    renderer: functionRendererPreviousLastWithOptionalVector,
});

register({
    type: 'histogram_percentile',
    addStrategy: addCombineStrategy,
    category: categories.Combine,
    params: [
        {name: "percentile", type: 'double', options: ['50', '80', '99', '99.9']},
        {name: "bucketLabel", quote: 'single', type: "string", options: ['bin', 'le', 'bucket']},
    ],
    defaultParams: ['90', 'bin'],
    renderer: functionRendererPreviousLastWithOptionalVector,
});

//Rank
register({
    type: 'top',
    addStrategy: addRankStrategy,
    category: categories.Rank,
    params: [
        {name: "count", type: "int", options: [1, 3, 5, 10, 20]},
        {name: "aggregator", quote: 'single', type: "string", options: ['max', 'min', 'avg', 'sum', 'last', 'count']}
        ],
    defaultParams: [3, 'last'],
    renderer: functionRendererPreviousLast,
});

register({
    type: 'bottom',
    addStrategy: addRankStrategy,
    category: categories.Rank,
    params: [
        {name: "count", type: "int", options: [1, 3, 5, 10, 20]},
        {name: "aggregator", quote: 'single', type: "string", options: ['max', 'min', 'avg', 'sum', 'last', 'count']}
    ],
    defaultParams: [3, 'last'],
    renderer: functionRendererPreviousLast,
});

// transformations
register({
    type: 'derivative',
    addStrategy: addTransformationStrategy,
    category: categories.Transformations,
    params: [],
    defaultParams: [],
    renderer: functionRenderer,
});

register({
    type: 'diff',
    addStrategy: addTransformationStrategy,
    category: categories.Transformations,
    params: [],
    defaultParams: [],
    renderer: functionRenderer,
});

register({
    type: 'integrate_fn',
    addStrategy: addTransformationStrategy,
    category: categories.Transformations,
    params: [],
    defaultParams: [],
    renderer: functionRenderer,
});

register({
    type: 'asap',
    addStrategy: addTransformationStrategy,
    category: categories.Transformations,
    params: [],
    defaultParams: [],
    renderer: functionRenderer,
});

register({
    type: 'drop_above',
    addStrategy: addTransformationStrategy,
    category: categories.Transformations,
    params: [
        {name: 'value', type: 'double', options: ['0', '10', '100', '1000']}
    ],
    defaultParams: ["100"],
    renderer: functionRenderer,
});

register({
    type: 'drop_below',
    addStrategy: addTransformationStrategy,
    category: categories.Transformations,
    params: [
        {name: 'value', type: 'double', options: ['0', '10', '100', '1000']}
    ],
    defaultParams: ["100"],
    renderer: functionRenderer,
});

register({
    type: 'drop_nan',
    addStrategy: addTransformationStrategy,
    category: categories.Transformations,
    params: [],
    defaultParams: [],
    renderer: functionRenderer,
});

register({
    type: 'replace_nan',
    addStrategy: addTransformationStrategy,
    category: categories.Transformations,
    params: [
        {name: 'value', type: 'double', options: ['0', '10', '100', '1000']}
    ],
    defaultParams: ["0"],
    renderer: functionRenderer,
});

register({
    type: 'non_negative_derivative',
    addStrategy: addTransformationStrategy,
    category: categories.Transformations,
    params: [],
    defaultParams: [],
    renderer: functionRenderer,
});

register({
    type: 'moving_avg',
    addStrategy: addTransformationStrategy,
    category: categories.Transformations,
    params: [{name: "duration", type: "interval", options: ['30s', '1m', '5m', '10m', '15m', '1h', '1d']}],
    defaultParams: ["1m"],
    renderer: functionRenderer,
});

register({
    type: 'moving_percentile',
    addStrategy: addTransformationStrategy,
    category: categories.Transformations,
    params: [
        {name: "duration", type: "interval", options: ['30s', '1m', '5m', '10m', '15m', '1h', '1d']},
        {name: 'nth', type: 'double', options: ['50', '80', '99', '99.9']},
    ],
    defaultParams: ["10m", '80'],
    renderer: functionRenderer,
});

register({
    type: 'moving_sum',
    addStrategy: addTransformationStrategy,
    category: categories.Transformations,
    params: [{name: "duration", type: "interval", options: ['30s', '1m', '5m', '10m', '15m', '1h', '1d']}],
    defaultParams: ["1m"],
    renderer: functionRenderer,
});


register({
    type: 'shift',
    addStrategy: addTransformationStrategy,
    category: categories.Transformations,
    params: [{name: "duration", type: "interval", options: ['5m', '10m', '15m', '1h', '1d', '1w']}],
    defaultParams: ["1d"],
    renderer: functionRenderer,
});



register({
    type: 'time',
    category: groupByTimeFunctions,
    params: [{name: "interval", type: "time", options: ['$__interval', '1s', '10s', '1m', '5m', '10m', '15m', '1h']}],
    defaultParams: ['$__interval'],
    renderer: functionRenderer,
});

register({
    type: 'fill',
    category: groupByTimeFunctions,
    params: [{name: "fill", type: "string", options: ['none', 'null', '0', 'previous', 'linear']}],
    defaultParams: ['null'],
    renderer: functionRenderer,
});

// predictions
register({
    type: 'holt_winters',
    addStrategy: addTransformationStrategy,
    category: categories.Predictors,
    params: [{name: "number", type: "int", options: [5, 10, 20, 30, 40]}, {
        name: "season",
        type: "int",
        options: [0, 1, 2, 5, 10]
    }],
    defaultParams: [10, 2],
    renderer: functionRenderer,
});

register({
    type: 'holt_winters_with_fit',
    addStrategy: addTransformationStrategy,
    category: categories.Predictors,
    params: [{name: "number", type: "int", options: [5, 10, 20, 30, 40]}, {
        name: "season",
        type: "int",
        options: [0, 1, 2, 5, 10]
    }],
    defaultParams: [10, 2],
    renderer: functionRenderer,
});

register({
    type: 'math',
    addStrategy: addMathStrategy,
    category: categories.Math,
    params: [{name: "expr", type: "string"}],
    defaultParams: [' / 100'],
    renderer: suffixRenderer,
});

register({
    type: 'alias',
    addStrategy: addAliasStrategy,
    category: categories.Aliasing,
    params: [{name: "name", quote: 'single', type: "string"}],
    defaultParams: ['alias'],
    renderMode: 'suffix',
    renderer: functionRenderer,
});


export default {
    create: createPart,
    getCategories: function () {
        return categories;
    }
};

