/* eslint-disable no-plusplus */
import { isWhitespace } from './Quoter';

class Parser {
  constructor(text, pos) {
    this._text = text;
    this._pos = pos;
  }

  static create(text) {
    return new Parser(text, 0);
  }

  copy() {
    return new Parser(this._text, this._pos);
  }

  hasNext() {
    return this._pos < this._text.length;
  }

  lookaheadChar() {
    return this._text[this._pos];
  }

  consumeChar() {
    return this._text[this._pos++];
  }

  consumeExpectedChar(expectedCh) {
    const actualCh = this._text[this._pos];

    if (actualCh !== expectedCh) {
      throw new Error(`Expected "${expectedCh}", but "${actualCh}": ${this._pos}`);
    }

    ++this._pos;
  }

  consumeWhitespaces() {
    while (this._pos < this._text.length && isWhitespace(this._text[this._pos])) {
      ++this._pos;
    }
  }

  consumeOneOf(variants) {
    for (let i = 0; i < variants.length; ++i) {
      const variant = variants[i];

      if (this._pos + variant.length < this._text.length) {
        const substr = this._text.substr(this._pos, variant.length);

        if (substr === variant) {
          this._pos += variant.length;
          return variant;
        }
      }
    }

    throw new Error(`Expected one of variants ${variants}: ${this._pos}`);
  }

  consumeWhile(predicate) {
    let value = '';

    while (this.hasNext() && predicate(this.lookaheadChar())) {
      value += this.consumeChar();
    }

    return value;
  }

  /**
   *
   * @param {RegExp} pattern
   * @return {string}
   */
  consumePattern(pattern) {
    if (!this.hasNext()) {
      return '';
    }

    let value = '';

    while (this.hasNext()) {
      const nextValue = value + this.lookaheadChar();

      if (!pattern.test(nextValue)) {
        return value;
      }

      this.consumeChar();

      value = nextValue;
    }

    return value;
  }
}

export default Parser;
