/* eslint-disable no-plusplus,no-param-reassign,no-continue */
import Parser from './Parser';
import Selector from './Selector';
import { consumeQuotedString, doubleQuote } from './Quoter';

const _SUPPORTED_OPERATORS = ['!==', '!=', '!~', '==', '=~', '='];
const _KEY_SYMBOL_PATTERN = /[-0-9a-zA-Z_]/;
const _SENSOR_NAME_PATTERN = /^[a-zA-Z_][a-zA-Z0-9_.]*$/;
const _FORBIDDEN_SYMBOLS = '"\'{}=!&,';

/**
 * Common class to parse/format/other manipulations with selectors
 */
class Selectors {
  /**
   *
   * @param {Parser} parser
   * @return {String} name selector
   * @private
   */
  static _consumeQuotedNameSelector(parser) {
    let nameSelector;

    const ch = parser.lookaheadChar();

    if (ch === '"' || ch === '\'') {
      nameSelector = consumeQuotedString(parser, []);
    } else {
      nameSelector = Selectors._consumeNameSelector(parser);
    }

    parser.consumeWhitespaces();

    return nameSelector;
  }

  static _consumeNameSelector(parser) {
    return parser.consumePattern(_SENSOR_NAME_PATTERN);
  }

  /**
   *
   * @param {Parser} parser
   * @return {Selector} selector
   * @private
   */
  static _consumeSelector(parser) {
    let key;

    parser.consumeWhitespaces();

    const first = parser.lookaheadChar();

    if (first === '\'' || first === '"') {
      key = consumeQuotedString(parser, _FORBIDDEN_SYMBOLS);
    } else {
      key = parser.consumeWhile(ch => _KEY_SYMBOL_PATTERN.test(ch));
    }

    parser.consumeWhitespaces();

    const op = parser.consumeOneOf(_SUPPORTED_OPERATORS);
    parser.consumeWhitespaces();

    const value = consumeQuotedString(parser, _FORBIDDEN_SYMBOLS);
    parser.consumeWhitespaces();

    if (op === '!=') {
      if (value === '*') {
        return Selector.absent(key);
      }

      if (value === '-') {
        return Selector.any(key);
      }
    }

    return new Selector(op, key, value);
  }

  static _formatNameSelector(nameSelector) {
    if (!nameSelector) {
      return '';
    }

    if (Selectors._SENSOR_NAME_PATTERN.test(nameSelector)) {
      return nameSelector;
    }

    return doubleQuote(nameSelector);
  }

  /**
   * Parses selectors text query
   * @param {String} text selectors query in text format
   * @return {Selectors} parsed selectors
   */
  static parse(text) {
    text = text.replace(/^\s+|\s+$/g, '');

    if (!text) {
      return new Selectors('', []);
    }

    const parser = Parser.create(text);

    return this.consumeSelectors(parser);
  }

  static consumeSelectors(parser) {
    const labelSelectors = [];

    parser.consumeWhitespaces();

    let nameSelector = '';
    let quoted = false;

    try {
      const copiedParser = parser.copy();
      nameSelector = Selectors._consumeQuotedNameSelector(copiedParser);
      copiedParser.consumeExpectedChar('{');
      parser._pos = copiedParser._pos;
      quoted = true;
    } catch (e) {
      nameSelector = '';
      if (parser.lookaheadChar() === '{') {
        parser.consumeExpectedChar('{');
        quoted = true;
      }
    }

    parser.consumeWhitespaces();

    if (quoted && parser.lookaheadChar() === '}') {
      parser.consumeExpectedChar('}');
      return new Selectors(nameSelector, labelSelectors);
    }

    labelSelectors.push(Selectors._consumeSelector(parser));

    while (parser.hasNext() && parser.lookaheadChar() !== '}') {
      parser.consumeExpectedChar(',');
      labelSelectors.push(Selectors._consumeSelector(parser));
    }

    parser.consumeWhitespaces();

    if (quoted) {
      parser.consumeExpectedChar('}');
    }

    return new Selectors(nameSelector, labelSelectors);
  }

  static of(nameSelector, ...labelSelectors) {
    return new Selectors(nameSelector, labelSelectors);
  }

  constructor(nameSelector, labelSelectors) {
    this._nameSelector = nameSelector;
    this._labelSelectors = labelSelectors;
  }

  getNameSelector() {
    return this._nameSelector;
  }

  getSelectors() {
    return this._labelSelectors;
  }

  format(reqiredBrackets = false) {
    const labelSelectorsPart = this._labelSelectors.map(selector => selector.format()).join(', ');

    if (!this._nameSelector) {
      return reqiredBrackets ? `{${labelSelectorsPart}}` : labelSelectorsPart;
    }

    const formattedNameSelector = Selectors._formatNameSelector(this._nameSelector);

    return `${formattedNameSelector}{${labelSelectorsPart}}`;
  }

  hasKey(key) {
    return this._labelSelectors.filter(selector => selector.getKey() === key).length > 0;
  }

  add(selector) {
    return new Selectors(this._nameSelector, [...this._labelSelectors, selector]);
  }

  addOverride(selector) {
    const filteredSelectors = this._labelSelectors.filter(s => s.getKey() !== selector.getKey());
    return new Selectors(this._nameSelector, [...filteredSelectors, selector]);
  }
}

export default Selectors;
