package ru.yandex.market.graphouse;

import java.io.File;
import java.io.FileWriter;
import java.io.IOException;
import java.nio.file.Files;
import java.nio.file.Paths;
import java.util.ArrayDeque;
import java.util.Collections;
import java.util.List;
import java.util.Map;
import java.util.concurrent.ArrayBlockingQueue;
import java.util.concurrent.ConcurrentHashMap;
import java.util.regex.Pattern;
import java.util.stream.Collectors;

import javax.annotation.Nullable;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import ru.yandex.solomon.staffOnly.annotations.ManagerMethod;

/**
 * @author alexlovkov
 */
public class PatternListFile {

    private static final Logger logger = LoggerFactory.getLogger(PatternListFile.class);
    private static final int QUEUE_CAPACITY = 10;

    private final ArrayDeque<Pattern> patterns;
    private final Map<String, StatisticMetric> lastIpsForMetric;
    private final String fileName;
    private final int ipsQueueSize;

    public PatternListFile(String fileName) {
        this(fileName, QUEUE_CAPACITY);
    }

    public PatternListFile(String fileName, int ipsQueueSize) {
        this.fileName = fileName;
        this.ipsQueueSize = ipsQueueSize;
        List<String> strings;
        try {
            strings = Files.readAllLines(Paths.get(fileName));
        } catch (IOException e) {
            // it's okay for first run on the machine
            logger.info("couldn't read list from file:{}", fileName, e);
            strings = Collections.emptyList();
        }
        this.patterns = strings.stream()
            .map(Pattern::compile)
            .collect(Collectors.toCollection(ArrayDeque::new));
        this.lastIpsForMetric = new ConcurrentHashMap<>(patterns.size());
        updateFile();
    }

    public boolean isMatch(String metric) {
        return isMatch(metric, null);
    }

    public boolean isMatch(String metric, @Nullable String ipAddress) {
        for (Pattern pattern : patterns) {
            if (pattern.matcher(metric).matches()) {
                if (ipAddress != null) {
                    updateIpStatistic(pattern, ipAddress);
                }
                return true;
            }
        }
        return false;
    }

    private void updateIpStatistic(Pattern pattern, String ipAddress) {
        lastIpsForMetric.compute(pattern.pattern(), (k, v) -> {
            if (v == null) {
                v = new StatisticMetric(new ArrayBlockingQueue<>(ipsQueueSize), 1);
            } else {
                if (v.getLastIpAddresses().size() >= ipsQueueSize) {
                    v.getLastIpAddresses().poll();
                }
                v.incrementCount();
            }
            v.getLastIpAddresses().offer(ipAddress);
            return v;
        });
    }

    @ManagerMethod
    public void addPattern(String pattern) {
        patterns.add(Pattern.compile(pattern));
        updateFile();
    }

    @ManagerMethod
    public void removePattern(String pattern) {
        patterns.removeIf(current -> pattern.equals(current.pattern()));
        lastIpsForMetric.remove(pattern);
        updateFile();
    }

    @ManagerMethod
    public void updatePattern(String oldPattern, String newPattern) {
        patterns.add(Pattern.compile(newPattern));
        patterns.removeIf(current -> oldPattern.equals(current.pattern()));
        lastIpsForMetric.remove(oldPattern);
        updateFile();
    }

    private void updateFile() {
        File file = new File(fileName);
        FileWriter writer = null;
        try {
            writer = new FileWriter(file, false);
            for (Pattern pattern : patterns) {
                writer.write(pattern.pattern() + System.lineSeparator());
            }
            logger.info("successfully updated:{}", fileName);
        } catch (IOException e) {
            logger.warn("couldn't write new pattern to the file:{}", fileName, e);
        } finally {
            if (writer != null) {
                try {
                    logger.info("close the file:{}", fileName);
                    writer.close();
                } catch (IOException e) {
                    logger.warn("couldn't close file:{}", fileName, e);
                }
            }
        }
    }

    /**
     * for test purpose only
     */
    public void clearFile() {
        try {
            Files.deleteIfExists(Paths.get(fileName));
        } catch (IOException e) {
            logger.warn("can't delete file:", fileName);
        }
    }

    @ManagerMethod
    public Map<String, StatisticMetric> getStatistic() {
        return lastIpsForMetric;
    }

    static class StatisticMetric {

        private long count;
        private final ArrayBlockingQueue<String> lastIpAddresses;

        StatisticMetric(ArrayBlockingQueue<String> lastIpAddresses, int count) {
            this.lastIpAddresses = lastIpAddresses;
            this.count = count;
        }

        long getCount() {
            return count;
        }

        void incrementCount() {
            this.count++;
        }

        ArrayBlockingQueue<String> getLastIpAddresses() {
            return lastIpAddresses;
        }

        @Override
        public String toString() {
            return "StatisticMetric{" +
                "count=" + count +
                ", queue=" + lastIpAddresses +
                '}';
        }
    }
}
