package ru.yandex.market.graphouse.monitoring;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.context.event.ContextClosedEvent;
import org.springframework.context.event.EventListener;
import org.springframework.stereotype.Controller;
import org.springframework.web.bind.annotation.RequestMapping;
import org.springframework.web.bind.annotation.ResponseBody;

import ru.yandex.market.graphouse.search.MetricSearch;
import ru.yandex.market.graphouse.stockpile.GraphouseStockpileClient;
import ru.yandex.monlib.metrics.registry.MetricRegistry;
import ru.yandex.solomon.selfmon.AvailabilityStatus;

/**
 * @author Maksim Leonov (nohttp@)
 */
@Controller
public class BalancerController {

    private static final Logger log = LoggerFactory.getLogger(BalancerController.class);

    @Autowired
    private GraphouseStockpileClient stockpileClient;
    @Autowired
    private MetricSearch metricSearch;

    private volatile boolean isShutdown = false;

    public BalancerController() {
        MetricRegistry.root().lazyGaugeInt64("balancerPing", () -> isOk() ? 1 : 0);
    }

    @RequestMapping("/balancer-ping")
    @ResponseBody
    public String checkStorage() {
        if (isOk()) {
            return "OK";
        } else {
            return "Error";
        }
    }

    private boolean isOk() {
        boolean dbOk = metricSearch.isInitialLoadComplete();
        AvailabilityStatus status = stockpileClient.getAvailability();

        if (!isShutdown && dbOk && Double.compare(status.getAvailability(), 1.0) >= 0) {
            return true;
        }

        log.info("Service is not available: " +
            "DB ready: " + dbOk + "; " +
            "stockpile: " + status.getAvailability() + "; " +
            "stockpile_available: " + status.getDetails() + "; " +
            "shutdown: " + isShutdown
        );
        return false;
    }

    @EventListener(ContextClosedEvent.class)
    public void shutdownHandler() {
        log.info("Detaching from balancer");
        isShutdown = true;
    }
}
