package ru.yandex.market.graphouse.retention;

import java.util.ArrayList;
import java.util.Collection;
import java.util.List;

import javax.annotation.Nonnull;

import org.springframework.stereotype.Component;

import ru.yandex.misc.lang.ByteUtils;
import ru.yandex.solomon.util.time.TimeProvider;
import ru.yandex.solomon.util.time.TimeProviderImpl;
import ru.yandex.stockpile.api.EDecimPolicy;

/**
 * @author Maksim Leonov (nohttp@)
 */
@Component
public class HardcodedRetentionManager implements RetentionManager {

    private final Retention[] retentions;
    private final Retention defaultRetention;
    private final TimeProvider timeProvider;

    public HardcodedRetentionManager() {
        timeProvider = new TimeProviderImpl();

        ArrayList<Retention> buffer = new ArrayList<>();

        buffer.addAll(buildRetentions("one_sec", 1_000));
        buffer.addAll(buildRetentions("five_sec", 5_000));
        buffer.addAll(buildRetentions("ten_sec", 10_000));
        buffer.addAll(buildRetentions("one_min", 60_000));

        buffer.addAll(buildRetentions("five_min", 300_000));
        buffer.addAll(buildRetentions("ten_min", 600_000));
        buffer.addAll(buildRetentions("half_hour", 1800_000));
        buffer.addAll(buildRetentions("one_hour", 3600_000));
        buffer.addAll(buildRetentions("one_day", 86400_000));

        // BOT-1871
        buffer.addAll(buildRetentions("one_week_del", 1_000, EDecimPolicy.POLICY_5_MIN_AFTER_7_DAYS));

        retentions = buffer.toArray(new Retention[buffer.size()]);

        // Lots of metrics without a prefix are usually one-min metrics. Anyway, all metrics have the same decim policy.
        defaultRetention = new Retention(
            m -> true,
            timeProvider,
            decimationPolicyIdUnderFiveMin(),
            60_000
        );
    }

    private Collection<Retention> buildRetentions(String part, int periodMillis) {
        EDecimPolicy policyId = ((periodMillis >= 300_000) ? decimationPolicyIdAboveFiveMin() : decimationPolicyIdUnderFiveMin());
        return buildRetentions(part, periodMillis, policyId);
    }

    private Collection<Retention> buildRetentions(String part, int periodMillis, EDecimPolicy policyId) {
        String prefix = part + ".";
        String suffix = "." + part;
        return List.of(
            new Retention(
                m -> m.startsWith(prefix),
                timeProvider,
                policyId,
                periodMillis
            ),
            new Retention(
                m -> m.endsWith(suffix),
                timeProvider,
                policyId,
                periodMillis
            )
        );
    }

    @Override
    public byte getRetentionIdForMetric(String metricName) {
        for (int x = 0; x < retentions.length; x++) {
            Retention retention = retentions[x];
            if (retention.matches(metricName)) {
                return ByteUtils.toByteExact(x);
            }
        }
        return ByteUtils.toByteExact(retentions.length);
    }

    @Nonnull
    @Override
    public Retention getRetentionById(short id) {
        if (id == retentions.length) {
            return defaultRetention;
        }
        return retentions[id];
    }

    private EDecimPolicy decimationPolicyIdUnderFiveMin() {
        return EDecimPolicy.POLICY_1_MIN_AFTER_1_MONTH_5_MIN_AFTER_3_MONTHS;
    }

    private EDecimPolicy decimationPolicyIdAboveFiveMin() {
//        there are some "bad" metrics which have actual data frequency higher than the one declared in metric names.
//        These metrics will be decimated with an 'avg' function,
//        and users accessing this averaged data will see something different from what they actually want.
//        I suppose it is a good idea to delay this decimation as much as possible, and for me it looks like
//        "after a month" policy can be acceptable for this data.

        return EDecimPolicy.POLICY_1_MIN_AFTER_1_MONTH_5_MIN_AFTER_3_MONTHS;
    }
}
