package ru.yandex.market.graphouse.retention;

import java.util.function.Predicate;

import javax.annotation.ParametersAreNonnullByDefault;

import ru.yandex.misc.lang.ShortUtils;
import ru.yandex.solomon.model.timeseries.decim.DecimPoliciesPredefined;
import ru.yandex.solomon.model.timeseries.decim.DecimPolicy;
import ru.yandex.solomon.model.timeseries.decim.DecimPolicyItem;
import ru.yandex.solomon.util.time.TimeProvider;
import ru.yandex.stockpile.api.EDecimPolicy;

/**
 * @author Maksim Leonov (nohttp@)
 */
@ParametersAreNonnullByDefault
public class Retention {
    private final TimeProvider timeProvider;

    private final DecimPolicy policy;
    private final short decimPolicyId;
    private final long defaultStepMillis;
    private final Predicate<String> metricMatcher;

    public Retention(Predicate<String> metricMatcher, TimeProvider timeProvider, EDecimPolicy decimPolicyId, long defaultStepMillis) {
        this.metricMatcher = metricMatcher;
        this.timeProvider = timeProvider;
        this.decimPolicyId = ShortUtils.toShortExact(decimPolicyId.getNumber());
        this.policy = DecimPoliciesPredefined.policyByNumber(this.decimPolicyId);
        this.defaultStepMillis = defaultStepMillis;
    }

    public final long getStepMillis(long fromMillis) {
        long result = defaultStepMillis;

        long ageMillis = timeProvider.nowMillis() - fromMillis;
        for (DecimPolicyItem item : policy.getItems()) {
            if (ageMillis < item.getAgeMillis()) {
                break;
            }
            // Resolution may _decrease_ while getting older data.
            // E.g.: default step is 'one day', but in one week, five minute decimation rule is applied.
            // But this method should return the largest step observed on the interval.
            result = Math.max(result, item.getStepMillis());
        }
        return result;
    }

    public DecimPolicy getPolicy() {
        return policy;
    }

    public boolean matches(String metricName) {
        return metricMatcher.test(metricName);
    }

    public short getPolicyId() {
        return decimPolicyId;
    }

    public long getDefaultStepMillis() {
        return defaultStepMillis;
    }
}
