package ru.yandex.market.graphouse.search;

import java.io.IOException;
import java.time.Instant;
import java.util.ArrayList;
import java.util.Collections;
import java.util.List;
import java.util.Queue;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.ConcurrentLinkedQueue;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.atomic.AtomicInteger;
import java.util.function.Consumer;
import java.util.stream.Collectors;
import java.util.stream.Stream;

import javax.annotation.Nonnull;
import javax.annotation.Nullable;
import javax.annotation.PostConstruct;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.context.event.ContextClosedEvent;
import org.springframework.context.event.EventListener;
import org.springframework.stereotype.Component;

import ru.yandex.market.graphouse.retention.RetentionManager;
import ru.yandex.market.graphouse.search.dao.MetricArray;
import ru.yandex.market.graphouse.search.dao.MetricRow;
import ru.yandex.market.graphouse.search.dao.MetricsDao;
import ru.yandex.market.graphouse.search.tree.MetricNameZip;
import ru.yandex.market.graphouse.search.tree.TreeStats;
import ru.yandex.market.graphouse.server.MetricLimbo;
import ru.yandex.market.graphouse.server.MetricValidator;
import ru.yandex.market.graphouse.stockpile.GraphouseStockpileIdGenerator;
import ru.yandex.misc.concurrent.CompletableFutures;
import ru.yandex.monlib.metrics.histogram.Histograms;
import ru.yandex.monlib.metrics.primitives.Counter;
import ru.yandex.monlib.metrics.primitives.Histogram;
import ru.yandex.monlib.metrics.primitives.Rate;
import ru.yandex.monlib.metrics.registry.MetricRegistry;
import ru.yandex.solomon.config.protobuf.graphite.storage.TMetricSearch;
import ru.yandex.stockpile.client.shard.StockpileMetricId;

/**
 * @author Dmitry Andreev <a href="mailto:AndreevDm@yandex-team.ru"/>
 * @date 07/04/15
 */
@Component
public class MetricSearch implements Runnable {

    private static final Logger log = LoggerFactory.getLogger(MetricSearch.class);

    private static final int BATCH_SIZE = 5_000;
    private static final int MAX_METRICS_PER_SAVE = 50_000;

    private final MetricsDao metricsDao;
    private final MetricValidator metricValidator;
    private final MetricLimbo metricLimbo;

    private final MetricTree metricTree;
    private final Queue<MetricDescription> updateQueue = new ConcurrentLinkedQueue<>();

    private final int saveIntervalSeconds;
    private final int loadUpdatedTtl;
    private final MetricSearchMetrics metricSearchMetrics;


    private int lastUpdatedTimestampSeconds = 0;
    /**
     * Задержка на запись, репликацию, синхронизацию
     */
    private final int updateDelaySeconds = 20;

    private volatile long startupTimeSeconds = 0;
    private volatile boolean initialLoadComplete = false;

    @Autowired
    public MetricSearch(
        GraphouseStockpileIdGenerator idGen,
        RetentionManager retentionManager,
        MetricsDao metricsDao,
        MetricValidator metricValidator,
        MetricLimbo metricLimbo,
        TMetricSearch metricConfig)
    {
        metricTree = new MetricTree(idGen, retentionManager);
        this.metricsDao = metricsDao;
        this.metricValidator = metricValidator;
        this.metricLimbo = metricLimbo;
        this.loadUpdatedTtl = metricConfig.getLoadUpdatedTtl();
        this.saveIntervalSeconds = metricConfig.getSaveIntervalSeconds();
        MetricRegistry metricRegistry = MetricRegistry.root();
        metricSearchMetrics = new MetricSearchMetrics(metricRegistry);
    }

    @PostConstruct
    public void afterPropertiesSet() {
        new Thread(this, "MetricSearch thread").start();
    }

    @EventListener(ContextClosedEvent.class)
    public void shutdownHandler() {
        log.info("Shutting down Metric search");
        saveUpdatedMetrics();
        log.info("Metric search stopped");
    }

    private void saveMetrics(List<MetricDescription> metrics) {
        if (metrics.isEmpty()) {
            return;
        }
        List<MetricRow> metricRows = metrics.stream().map(metricDescription -> {
            String metricName = metricDescription.getName() + (metricDescription.isDir() ? "." : "");
            int statusId = metricDescription.getStatus().getId();
            int shardId;
            long localId;
            if (metricDescription.isDir()) {
                shardId = 0;
                localId = 0;
            } else {
                StockpileMetricId stockpileMetricId = metricDescription.getStockpileId();
                shardId = stockpileMetricId.getShardId();
                localId = stockpileMetricId.getLocalId();
            }
            return MetricRow.of(metricName, statusId, Instant.now(), shardId, localId);
        }).collect(Collectors.toList());
        CompletableFutures.join(metricsDao.saveMetrics(metricRows));
    }

    private CompletableFuture<Void> loadAllMetrics() {
        log.info("Loading all metric names from db...");
        long startTime = System.currentTimeMillis();

        return metricsDao.loadAllMetrics(this::insertBatchWithoutValidation)
            .thenAccept(metricRows -> {
                startupTimeSeconds = TimeUnit.MILLISECONDS.toSeconds(System.currentTimeMillis() - startTime);
                metricSearchMetrics.recordStartupTimeSeconds(startupTimeSeconds);
                log.info("Loading metrics complete. Total {} metrics for {} seconds",
                    metricRows, startupTimeSeconds);
            });
    }

    private CompletableFuture<Void> loadUpdatedMetrics(int startTimestampSeconds) {
        log.info("Loadinng updated metric names from db... startTimestampSeconds:{}", startTimestampSeconds);
        long startTime = System.currentTimeMillis();

        return metricsDao.loadNewMetrics(this::insertBatch, startTimestampSeconds)
            .thenAccept((metricRows) -> {
                long operationTime = System.currentTimeMillis() - startTime;
                log.info("Loading lastUpdatedMetrics complete. Total {} metrics in {} ms",
                    metricRows, operationTime);
                metricSearchMetrics.recordHistogramLoadUpdated(operationTime);
                metricSearchMetrics.addUpdatesReadFromDb(metricRows);
                metricLimbo.onDbUpdateFetched();
            });
    }

    public Stream<MetricDescription> allMetrics() {
        return metricTree.allMetrics();
    }

    public boolean isInitialLoadComplete() {
        return initialLoadComplete;
    }

    public TreeStats stats() {
        return metricTree.stats();
    }

    private void insertBatch(MetricArray batch) {
        PredefinedStockpileId idGenerator = new PredefinedStockpileId();
        for (int index = 0; index < batch.size(); index++) {
            MetricResponseStatus status = metricValidator.validate(batch.getName(index), true, null);
            if (status != MetricResponseStatus.OK) {
                log.warn("Invalid metric in db:{} status:{}", batch.getName(index), status);
            } else {
                idGenerator.set(batch.getShardId(index), batch.getLocalId(index));
                modifyMetric(batch.getName(index), batch.getStatus(index), idGenerator);
            }
        }
    }

    private void insertBatchWithoutValidation(MetricArray batch) {
        PredefinedStockpileId idGenerator = new PredefinedStockpileId();
        for (int index = 0; index < batch.size(); index++) {
            idGenerator.set(batch.getShardId(index), batch.getLocalId(index));
            modifyMetric(batch.getName(index), batch.getStatus(index), idGenerator);
        }
    }

    private void modifyMetric(String metric, MetricTreeStatus status, GraphouseStockpileIdGenerator idSource) {
        try {
            metricTree.modify(metric, status, idSource);
        } catch (MetricAddException e) {
            log.warn("Invalid metric in db: " + metric);
        }
    }

    private boolean saveUpdatedMetrics() {
        if (updateQueue.isEmpty()) {
            log.info("No new metric names to save");
            return false;
        }
        long saveUpdatedMetricsStartTime = System.currentTimeMillis();
        log.info("Saving new metric names to db. Current count: " + updateQueue.size());
        List<MetricDescription> metrics = new ArrayList<>();
        MetricDescription metric;
        while (metrics.size() < MAX_METRICS_PER_SAVE && (metric = updateQueue.poll()) != null) {
            if (metric instanceof MetricNameZip) {
                MetricNameZip metricName = (MetricNameZip) metric;
                if (metricName.getStatus() == MetricTreeStatus.SIMPLE && metricName.self.isFromDb()) {
                    // This newly created metric was already updated from the db and there is no status change awaiting.
                    // Thus there is no need to write it to the DB.
                    continue;
                }
            }
            metrics.add(metric);
        }

        int processed = 0;
        try {
            while (processed < metrics.size()) {
                List<MetricDescription> batch = metrics.subList(processed, Math.min(metrics.size(), processed + BATCH_SIZE));

                long startMillis = System.currentTimeMillis();
                saveMetrics(batch);
                long dtMillis = System.currentTimeMillis() - startMillis;
                long speed = batch.size() * 1000 / Math.max(1, dtMillis);
                log.info("Saved a batch of " + batch.size() + " metric names in " + dtMillis + "ms: " + speed + " metrics/sec");
                metricSearchMetrics.addUpdatesStoredToDb(batch.size());

                processed += BATCH_SIZE;
                if (Thread.currentThread().isInterrupted()) {
                    throw new InterruptedException();
                }
            }
            long operationTime = System.currentTimeMillis() - saveUpdatedMetricsStartTime;
            log.info("Saved all the " + metrics.size() + " metric names for the current iteration for:{}ms",
                operationTime);
            metricSearchMetrics.recordHistogramSaveUpdatedMetricsTime(operationTime);
        } catch (Exception e) {
            log.error("Failed to save metrics to db", e);
            metricSearchMetrics.incSaveKikimrFails();
            updateQueue.addAll(metrics.subList(processed, metrics.size()));
        }
        return metrics.size() == MAX_METRICS_PER_SAVE;
    }

    @Override
    public void run() {
        while (!Thread.interrupted()) {
            try {
                log.info(
                    "Actual metrics count = " + metricTree.metricCount() + ", dir count: " + metricTree.dirCount()
                );
                CompletableFuture<Void> loadFuture = loadNewMetrics();
                boolean hasMoreMetrics = loadFuture.thenApply(e -> saveUpdatedMetrics()).join();
                if (hasMoreMetrics) {
                    continue;
                }
            } catch (Exception e) {
                log.error("Failed to update metric search", e);
            }
            if (TimeUnit.MILLISECONDS.toSeconds(System.currentTimeMillis()) - updateDelaySeconds - lastUpdatedTimestampSeconds < saveIntervalSeconds) {
                try {
                    log.info("MetricSearch sleep for {} seconds", saveIntervalSeconds);
                    Thread.sleep(TimeUnit.SECONDS.toMillis(saveIntervalSeconds));
                } catch (InterruptedException ignored) {
                }
            }
        }
    }

    private CompletableFuture<Void> loadNewMetrics() {
        int timeSeconds = (int) (TimeUnit.MILLISECONDS.toSeconds(System.currentTimeMillis())) - updateDelaySeconds;

        CompletableFuture<Void> result;
        if (lastUpdatedTimestampSeconds == 0) {
            result = loadAllMetrics().thenAccept(ignore -> initialLoadComplete = true);
        } else {
            // Actual saves can be quite slow; we don't want to miss any data,
            // so we download data for [now-ttl, now] period
            result = loadUpdatedMetrics(lastUpdatedTimestampSeconds - loadUpdatedTtl);
        }
        return result.thenAccept(e -> lastUpdatedTimestampSeconds = timeSeconds);
    }

    @Nonnull
    public MetricDescription add(String metric) throws MetricAddException {
        long currentTimeMillis = System.currentTimeMillis();

        MetricDescription metricDescription = metricTree.add(metric);
        if (metricDescription.getUpdateTimeMillis() >= currentTimeMillis) {
            updateQueue.add(metricDescription);
        }
        return metricDescription;
    }

    public int multiModify(String query, final MetricTreeStatus status, final Appendable result) throws IOException {
        final StringBuilder metricBuilder = new StringBuilder();
        final AtomicInteger count = new AtomicInteger();

        metricTree.search(query, new Appendable() {
            @Override
            public Appendable append(CharSequence csq) {
                metricBuilder.append(csq);
                return this;
            }

            @Override
            public Appendable append(CharSequence csq, int start, int end) {
                metricBuilder.append(csq, start, end);
                return this;
            }

            @Override
            public Appendable append(char c) throws IOException {
                if (c == '\n') {
                    modify(metricBuilder.toString(), status);
                    if (result != null) {
                        result.append(metricBuilder).append('\n');
                    }
                    metricBuilder.setLength(0);
                    count.incrementAndGet();
                } else {
                    metricBuilder.append(c);
                }
                return this;
            }
        });
        return count.get();
    }

    public void modify(String metric, MetricTreeStatus status) {
        modify(Collections.singletonList(metric), status, ModifyType.UPDATE_DB);
    }

    public void modify(List<String> metrics, MetricTreeStatus status, ModifyType modifyType) {
        if (metrics == null || metrics.isEmpty()) {
            return;
        }
        if (status == MetricTreeStatus.SIMPLE) {
            throw new IllegalStateException("Cannon modify to SIMPLE status");
        }
        long currentTimeMillis = System.currentTimeMillis();
        List<MetricDescription> metricDescriptions = new ArrayList<>();
        for (String metric : metrics) {
            if (metricValidator.validate(metric, true, null) != MetricResponseStatus.OK) {
                log.warn("Wrong metric to modify: " + metric);
                continue;
            }
            try {
                MetricDescription metricDescription = metricTree.modify(metric, status);
                if (metricDescription.getUpdateTimeMillis() >= currentTimeMillis) {
                    metricDescriptions.add(metricDescription);
                }
            } catch (MetricAddException _ignored) {
                // nop
            }
        }
        if (modifyType == ModifyType.UPDATE_DB) {
            saveMetrics(metricDescriptions);
        }
        if (metrics.size() == 1) {
            log.info("Updated metric '" + metrics.get(0) + "', status: " + status.name());
        } else {
            log.info("Updated " + metrics.size() + " metrics, status: " + status.name());
        }
    }

    public void search(String query, Appendable result) throws IOException {
        metricTree.search(query, result);
    }

    public void searchAllMetrics(String query, Consumer<String> consumer) {
        metricTree.searchAllMetrics(query, consumer);
    }

    @Nullable
    public MetricNameZip findExistingMetric(String metric) {
        return metricTree.findExistingMetric(metric);
    }

    private class MetricSearchMetrics {

        private final Histogram histogramLoadUpdated;
        private final Histogram histogramSaveUpdatedMetricsTime;
        private final Counter startupTimeSecondsCounter;
        private final Rate updatesReadFromDb;
        private final Rate updatesStoredToDb;
        private final Counter saveKikimrFails;


        MetricSearchMetrics(MetricRegistry metricRegistry) {
            metricRegistry.lazyGaugeInt64("MetricSearch.totalMetrics", metricTree::metricCount);
            metricRegistry.lazyGaugeInt64("MetricSearch.totalDirs", metricTree::dirCount);
            this.histogramLoadUpdated = metricRegistry
                .histogramRate("MetricSearch.loadUpdated", Histograms.exponential(13, 2, 16));
            this.histogramSaveUpdatedMetricsTime =
                metricRegistry.histogramRate("MetricSearch.saveUpdatedMetricsTime", Histograms.exponential(13, 2, 16));
            this.startupTimeSecondsCounter = metricRegistry.counter("MetricSearch.startupTimeSeconds");
            this.updatesReadFromDb = metricRegistry.rate("MetricSearch.updatesReadFromDb");
            this.updatesStoredToDb = metricRegistry.rate("MetricSearch.updatesStoredToDb");
            this.saveKikimrFails = metricRegistry.counter("MetricSearch.saveKikimrFails");
        }

        void recordStartupTimeSeconds(long startupTimeSeconds) {
            startupTimeSecondsCounter.add(startupTimeSeconds);
        }

        void recordHistogramLoadUpdated(long operationTime) {
            histogramLoadUpdated.record(operationTime);
        }

        void recordHistogramSaveUpdatedMetricsTime(long operationTime) {
            histogramSaveUpdatedMetricsTime.record(operationTime);
        }

        void addUpdatesReadFromDb(int metricRows) {
            updatesReadFromDb.add(metricRows);
        }

        void addUpdatesStoredToDb(int metricRows) {
            updatesStoredToDb.add(metricRows);
        }

        void incSaveKikimrFails() {
            saveKikimrFails.inc();
        }
    }

    private static class PredefinedStockpileId implements GraphouseStockpileIdGenerator {
        private int shardId;
        private long localId;

        public PredefinedStockpileId() {
        }

        public void set(int shardId, long localId) {
            this.shardId = shardId;
            this.localId = localId;
        }

        @Override
        public StockpileMetricId generateMetricId() {
            return new StockpileMetricId(shardId, localId);
        }

        @Override
        public boolean isStockpileIdFromDatabase() {
            return true;
        }
    }
}
