package ru.yandex.market.graphouse.search;

import java.io.IOException;
import java.nio.file.FileSystems;
import java.nio.file.Path;
import java.nio.file.PathMatcher;
import java.util.ArrayList;
import java.util.List;
import java.util.concurrent.atomic.AtomicLong;
import java.util.function.Consumer;
import java.util.regex.PatternSyntaxException;
import java.util.stream.Stream;

import javax.annotation.Nonnull;
import javax.annotation.Nullable;

import com.google.common.base.CharMatcher;
import org.apache.commons.lang3.StringUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import ru.yandex.market.graphouse.retention.RetentionManager;
import ru.yandex.market.graphouse.search.tree.Dir;
import ru.yandex.market.graphouse.search.tree.DirZip;
import ru.yandex.market.graphouse.search.tree.MetricBase;
import ru.yandex.market.graphouse.search.tree.MetricBaseZip;
import ru.yandex.market.graphouse.search.tree.MetricDescriptionImpl;
import ru.yandex.market.graphouse.search.tree.MetricNameZip;
import ru.yandex.market.graphouse.search.tree.TreeStats;
import ru.yandex.market.graphouse.search.util.MetricPath;
import ru.yandex.market.graphouse.stockpile.GraphouseStockpileIdGenerator;

/**
 * @author Dmitry Andreev <a href="mailto:AndreevDm@yandex-team.ru"/>
 * @date 07/04/15
 */
public class MetricTree {
    private static final Logger log = LoggerFactory.getLogger(MetricTree.class);

    private static final CharMatcher EXPRESSION_MATCHER = CharMatcher.anyOf("*?[]{}");
    private final Dir root = new Dir();

    private final GraphouseStockpileIdGenerator idGen;
    private final RetentionManager retentionManager;

    private final AtomicLong metricsCount = new AtomicLong();
    private final AtomicLong dirsCount = new AtomicLong();

    MetricTree(GraphouseStockpileIdGenerator idGen, RetentionManager retentionManager) {
        this.idGen = idGen;
        this.retentionManager = retentionManager;
    }

    private DirZip rootZip() {
        return new DirZip(root);
    }

    public void searchAllMetrics(String query, Consumer<String> consumer) {
        String[] levels = StringUtils.split(query,'.');
        searchAllMetrics(rootZip(), levels, 0, consumer);
    }

    public void search(String query, Appendable result) throws IOException {
        String[] levels = StringUtils.split(query,'.');
        search(rootZip(), levels, 0, result);
    }

    /**
     * Рекурсивный метод для получения списка метрик внутри дерева.
     *
     * @param parentDir  внутри какой директории ищем
     * @param levels     узлы дерева, каждый может быть задан явно или паттерном, используя *?[]{}
     *                   Пример: five_min.abo-main.timings-method.*.0_95
     * @param levelIndex индекс текущего узла
     * @param result
     * @throws IOException
     */
    private void search(DirZip parentDir, String[] levels, int levelIndex, Appendable result) throws IOException {
        if (!parentDir.self.visible()) {
            return;
        }
        boolean isLast = (levelIndex == levels.length - 1);
        String level = levels[levelIndex];
        boolean isPattern = containsExpressions(level);

        if (!isPattern) {
            if (isLast) {
                appendSimpleResult(parentDir, level, result);
            } else {
                DirZip dir = parentDir.getDir(level);
                if (dir != null) {
                    search(dir, levels, levelIndex + 1, result);
                }
            }
        } else if (level.equals("*")) {
            if (isLast) {
                appendAllResult(parentDir, result);
            } else {
                for (DirZip dir : parentDir.listDirs()) {
                    search(dir, levels, levelIndex + 1, result);
                }
            }
        } else {
            PathMatcher pathMatcher = createPathMatcher(level);
            if (pathMatcher == null) {
                return;
            }
            if (isLast) {
                appendPatternResult(parentDir, pathMatcher, result);
            } else {
                for (DirZip dir : parentDir.listDirs()) {
                    if (matches(pathMatcher, dir.self.getLastNameChunk())) {
                        search(dir, levels, levelIndex + 1, result);
                    }
                }
            }
        }
    }

    /**
     * get all metrics in the tree by pattern
     * almost copy-paste from search, but don't stop after we found dir which contains all nodes
     * recursively continue append all leafs to the result
     *
     * @param parentDir  directory of the tree
     * @param levels     nodes of the tree, every node can be metric or pattern with *?[]{}
     *                   example: five_min.abo-main.timings-method.*.0_95
     * @param levelIndex index of the current node
     * @param consumer
     */
    private void searchAllMetrics(
        DirZip parentDir,
        String[] levels,
        int levelIndex,
        Consumer<String> consumer)
    {
        if (!parentDir.self.visible()) {
            return;
        }
        boolean isLast = (levelIndex == levels.length - 1);
        String level = levels[levelIndex];
        boolean isPattern = containsExpressions(level);

        if (!isPattern) {
            if (isLast) {
                appendSimpleResult(parentDir, level, consumer);
            } else {
                DirZip dir = parentDir.getDir(level);
                if (dir != null) {
                    searchAllMetrics(dir, levels, levelIndex + 1, consumer);
                }
            }
        } else if (level.equals("*")) {
            for (DirZip dir : parentDir.listDirs()) {
                if (isLast) {
                    searchAllMetrics(dir, levels, levelIndex, consumer);
                } else {
                    searchAllMetrics(dir, levels, levelIndex + 1, consumer);
                }
            }
            appendAllMetrics(parentDir, consumer);
        } else {
            PathMatcher pathMatcher = createPathMatcher(level);
            if (pathMatcher == null) {
                return;
            }
            for (DirZip dir : parentDir.listDirs()) {
                if (matches(pathMatcher, dir.self.getLastNameChunk())) {
                    if (isLast) {
                        searchAllMetrics(dir, levels, levelIndex, consumer);
                    } else {
                        searchAllMetrics(dir, levels, levelIndex + 1, consumer);
                    }
                }
            }
            appendAllMetrics(parentDir, consumer);
        }
    }

    @Nullable
    static PathMatcher createPathMatcher(String globPattern) {
        try {
            return FileSystems.getDefault().getPathMatcher("glob:" + globPattern);
        } catch (PatternSyntaxException e) {
            return null;
        }
    }

    static boolean matches(PathMatcher pathMatcher, final String fileName) {
        Path x = new MetricPath(fileName);
        return pathMatcher.matches(x);
    }

    long metricCount() {
        return metricsCount.get();
    }

    long dirCount() {
        return dirsCount.get();
    }

    private void appendSimpleResult(DirZip parentDir, String name, Appendable result) throws IOException {
        appendResult(parentDir.getDir(name), result);
        appendResult(parentDir.getMetric(name), result);
    }

    private void appendSimpleResult(DirZip parentDir, String name, Consumer<String> consumer) {
        appendResult(parentDir.getDir(name), consumer);
        appendResult(parentDir.getMetric(name), consumer);
    }

    // TODO: delete, use methods with consumer
    private void appendAllResult(DirZip parentDir, Appendable result) throws IOException {
        for (DirZip dir : parentDir.listDirs()) {
            appendResult(dir, result);
        }
        for (MetricNameZip metric : parentDir.listMetrics()) {
            appendResult(metric, result);
        }
    }

    private void appendAllMetrics(DirZip parentDir, Consumer<String> consumer) {
        for (MetricNameZip metric : parentDir.listMetrics()) {
            appendResult(metric, consumer);
        }
    }

    // TODO: delete, use methods with consumer
    private void appendPatternResult(DirZip parentDir, PathMatcher pathMatcher, Appendable result) throws IOException {
        for (DirZip dir : parentDir.listDirs()) {
            if (dir.self.visible() && matches(pathMatcher, dir.self.getLastNameChunk())) {
                appendResult(dir, result);
            }
        }
        for (MetricNameZip metricName : parentDir.listMetrics()) {
            if (metricName.self.visible() && matches(pathMatcher, metricName.self.getLastNameChunk())) {
                appendResult(metricName, result);
            }
        }
    }

    // TODO: delete, use methods with consumer
    private void appendResult(@Nullable DirZip dir, Appendable result) throws IOException {
        if (dir != null && dir.self.visible()) {
            appendDir(dir, result);
            result.append('\n');
        }
    }

    // TODO: delete, use methods with consumer
    private void appendResult(@Nullable MetricNameZip metric, Appendable result) throws IOException {
        if (metric != null && metric.self.visible()) {
            appendDir(metric.getParent(), result);
            result.append(metric.self.getLastNameChunk()).append('\n');
        }
    }

    private void appendResult(@Nullable DirZip dir, Consumer<String> consumer) {
        if (dir != null && dir.self.visible()) {
            consumer.accept(appendDir(dir).toString());
        }
    }

    private void appendResult(@Nullable MetricNameZip metric, Consumer<String> consumer) {
        if (metric != null && metric.self.visible()) {
            StringBuilder dir = appendDir(metric.getParent());
            consumer.accept(dir.append(metric.self.getLastNameChunk()).toString());
        }
    }

    // TODO: delete, use methods with consumer
    private void appendDir(DirZip dir, Appendable result) throws IOException {
        if (dir.isRoot()) {
            return;
        }
        appendDir(dir.getParent(), result);
        result.append(dir.self.getLastNameChunk()).append('.');
    }

    private StringBuilder appendDir(DirZip dir) {
        if (dir.isRoot()) {
            return new StringBuilder();
        }
        StringBuilder sb = appendDir(dir.getParent());

        return sb.append(dir.self.getLastNameChunk()).append('.');
    }

    @Nonnull
    public MetricBaseZip<? extends MetricBase> add(String metric) throws MetricAddException {
        return modify(metric, MetricTreeStatus.SIMPLE).asZip();
    }

    @Nullable
    public MetricNameZip findExistingMetric(String metric) {
        boolean isDir = metric.charAt(metric.length() - 1) == '.';
        if (isDir) {
            throw new IllegalStateException();
        }

        String[] levels = metric.split("\\.");
        try {
            List<Dir> parents = findParent(levels, false);
            if (parents.isEmpty()) {
                return null;
            }
            String name = levels[levels.length - 1];
            return DirZip.of(parents).getMetric(name);
        } catch (MetricAddException e) {
            return null;
        }
    }

    /**
     * Создает или изменяет статус метрики или целой директории.
     *
     * @param metric если заканчивается на '.' , то директория
     * @param status
     * @return MetricDescription
     */
    @Nonnull
    MetricDescription modify(String metric, MetricTreeStatus status) throws MetricAddException {
        return modify(metric, status, idGen);
    }

    @Nonnull
    private List<Dir> findParent(String[] levels, boolean createIfNotExists) throws MetricAddException {
        List<Dir> parents = new ArrayList<>(levels.length - 1);
        var dir = this.root;
        parents.add(dir);
        for (int i = 0; i < levels.length; i++) {
            boolean isLast = (i == levels.length - 1);
            if (dir.getStatus() == MetricTreeStatus.BAN) {
                throw MetricAddException.problem(MetricResponseStatus.REJECTED_BANNED);
            }
            String level = levels[i];
            if (!isLast) {
                if (createIfNotExists) {
                    dir = dir.getOrCreateDir(this, level);
                } else {
                    dir = dir.getDir(level);
                    if (dir == null) {
                        return List.of();
                    }
                }

                if (parents.get(parents.size() - 1).visible() != dir.visible()) {
                    updatePathVisibility(DirZip.of(parents));
                }
                parents.add(dir);
            } else {
                return parents;
            }
        }
        throw new IllegalStateException();
    }

    @Nonnull
    MetricDescription modify(String metric, MetricTreeStatus status, GraphouseStockpileIdGenerator idSource) throws MetricAddException {
        String[] levels = metric.split("\\.");
        var dirs = findParent(levels, true);

        boolean isDir = metric.charAt(metric.length() - 1) == '.';
        String level = levels[levels.length - 1];

        MetricDescription metricDescription = modify(dirs, level, metric, isDir, status, idSource);
        if (status.visible() != dirs.get(dirs.size() - 1).visible()) {
            updatePathVisibility(DirZip.of(dirs));
        }
        return metricDescription;
    }

    private MetricDescription modify(List<Dir> parents, String name, String fullName, boolean isDir, MetricTreeStatus status, GraphouseStockpileIdGenerator idSource) {
        if (parents.size() == 1) {
            throw new IllegalArgumentException("Disallowed modify second level " + fullName);
        }
        var parent = parents.get(parents.size() - 1);
        if (isDir) {
            var dir = parent.getOrCreateDir(this, name);
            dir.setStatus(selectStatus(dir.getStatus(), status));
            return new MetricDescriptionImpl<>(parents, dir);
        } else {
            var metric = parent.getOrCreateMetric(this, name, fullName, idSource, retentionManager);
            metric.setStatus(selectStatus(metric.getStatus(), status));
            return new MetricDescriptionImpl<>(parents, metric);
        }
    }

    /**
     * Если все метрики в директории скрыты, то пытаемся скрыть её {@link MetricTreeStatus#AUTO_HIDDEN}
     * Если для директории есть хоть одна открытая метрика, то пытаемся открыть её {@link MetricTreeStatus#SIMPLE}
     *
     * @param dir
     */
    private void updatePathVisibility(DirZip dir) {
        if (dir.isRoot()) {
            return;
        }
        MetricTreeStatus newStatus = selectStatus(
            dir.self.getStatus(),
            hasVisibleChildren(dir) ? MetricTreeStatus.SIMPLE : MetricTreeStatus.AUTO_HIDDEN
        );
        if (dir.self.getStatus() != newStatus) {
            dir.self.setStatus(newStatus);
            updatePathVisibility(dir.getParent());
        }
    }

    private boolean hasVisibleChildren(DirZip dir) {
        for (DirZip child : dir.listDirs()) {
            if (child.self.visible()) {
                return true;
            }
        }
        for (MetricNameZip child : dir.listMetrics()) {
            if (child.self.visible()) {
                return true;
            }
        }
        return false;
    }

    /**
     * Возвращаем новый статус при изменении метрики, учитывая граф возможных переходов.
     *
     * @param oldStatus
     * @param newStatus
     * @return
     */
    private MetricTreeStatus selectStatus(MetricTreeStatus oldStatus, MetricTreeStatus newStatus) {
        List<MetricTreeStatus> restricted = MetricTreeStatus.RESTRICTED_GRAPH_EDGES.get(oldStatus);
        return restricted == null || !restricted.contains(newStatus) ? newStatus : oldStatus;
    }

    static boolean containsExpressions(String metric) {
        return EXPRESSION_MATCHER.matchesAnyOf(metric);
    }

    Stream<MetricDescription> allMetrics() {
        return rootZip().allMetrics();
    }

    public void registerNewDir() {
        dirsCount.incrementAndGet();
    }

    public void registerNewMetric() {
        metricsCount.incrementAndGet();
    }

    TreeStats stats() {
        TreeStats stats = new TreeStats();
        root.updateStats(stats);
        return stats;
    }
}
