package ru.yandex.market.graphouse.search;

import java.util.Arrays;
import java.util.Collections;
import java.util.EnumMap;
import java.util.List;
import java.util.Map;

/**
 * @author Dmitry Andreev <a href="mailto:AndreevDm@yandex-team.ru"/>
 * @date 08/06/15
 */
public enum MetricTreeStatus {
    /**
     * Статус по умолчанию при создании директории/метрики.
     */
    SIMPLE(0),
    /**
     * Если директория (метрика) забанена, то
     * - директория и все метрики в ней (метрика) перестаёт находиться в поиске (а следовательно и в графите)
     * - значения метрик в директории (метрики) перестают приниматься и писаться в графит
     * <p/
     * Чтобы открыть деректорию(метрику), необходимо явно перевести в {@link #APPROVED}
     */
    BAN(1),
    APPROVED(2),
    /**
     * Если директория(метрика) скрыта, то
     * - директория и все метрики в ней (метрика) перестаёт находиться в поиске (а следовательно и в графите)
     * - как только появится новое значение метрика и все родительские директории будут открыты
     * <p/>
     */
    HIDDEN(3),
    /**
     * Директория автоматически скрывается, если все её дочерние директории и метрики не видимы {@link #visible}
     * Как только появится новое значение для дочерней метрики, директория будет открыта {@link #SIMPLE}
     * <p/>
     * TODO: Реализовать аналогичное поведение в Stockpile нетривиально, поэтому это пока не поддерживается.
     * Изначальную реализацию см. в AutoHideService в оригинальном репозитории.
     */
    AUTO_HIDDEN(4);

    public static final MetricTreeStatus[] values = values();

    public static final Map<MetricTreeStatus, List<MetricTreeStatus>> RESTRICTED_GRAPH_EDGES = new EnumMap<>(
        MetricTreeStatus.class
    );

    static {
        RESTRICTED_GRAPH_EDGES.put(MetricTreeStatus.BAN, Arrays.asList(MetricTreeStatus.SIMPLE, MetricTreeStatus.AUTO_HIDDEN));
        RESTRICTED_GRAPH_EDGES.put(MetricTreeStatus.HIDDEN, Collections.singletonList(MetricTreeStatus.AUTO_HIDDEN));
        RESTRICTED_GRAPH_EDGES.put(MetricTreeStatus.APPROVED, Arrays.asList(MetricTreeStatus.SIMPLE, MetricTreeStatus.AUTO_HIDDEN));
    }

    private final int id;

    MetricTreeStatus(int id) {
        this.id = id;
    }

    public int getId() {
        return id;
    }

    /**
     * Если <code>false</code>, то в поиске не будет отдаваться данная метрика (ни одна метрика из данной директории).
     */
    public boolean visible() {
        switch (this) {
            case SIMPLE:
            case APPROVED:
                return true;
            case BAN:
            case HIDDEN:
            case AUTO_HIDDEN:
                return false;
            default:
                throw new IllegalStateException();
        }
    }

    public boolean handmade() {
        switch (this) {
            case APPROVED:
            case BAN:
            case HIDDEN:
                return true;
            case SIMPLE:
            case AUTO_HIDDEN:
                return false;
            default:
                throw new IllegalStateException();
        }
    }

    public static MetricTreeStatus forId(int id) {
        return values[id];
    }
}
