package ru.yandex.market.graphouse.search.dao;

import java.time.Duration;
import java.util.Optional;

import com.yandex.ydb.table.SchemeClient;
import com.yandex.ydb.table.TableClient;
import org.springframework.context.annotation.Bean;
import org.springframework.context.annotation.Configuration;

import ru.yandex.market.graphouse.search.dao.ydb.YdbMetricsDao;
import ru.yandex.monlib.metrics.MetricSupplier;
import ru.yandex.monlib.metrics.registry.MetricRegistry;
import ru.yandex.solomon.config.TimeConverter;
import ru.yandex.solomon.config.protobuf.graphite.storage.TKikimrConfig;
import ru.yandex.solomon.config.thread.ThreadPoolProvider;
import ru.yandex.solomon.secrets.SecretProviders;
import ru.yandex.solomon.selfmon.mon.DaoMetricsProxy;
import ru.yandex.solomon.util.actors.PingActorRunner;
import ru.yandex.solomon.ydb.YdbClients;

/**
 * @author alexlovkov
 */
@Configuration
public class DaoContext implements AutoCloseable {

    private final TKikimrConfig config;
    private final ThreadPoolProvider threads;
    private final YdbClients ydbClients;

    public DaoContext(TKikimrConfig config, ThreadPoolProvider threads) {
        this.config = config;
        this.threads = threads;
        this.ydbClients = new YdbClients("KikimrConfig.KikimrClientConfig",
                config.getKikimrClientConfig(),
                threads,
                MetricRegistry.root(),
                Optional.empty(),
                SecretProviders.empty());
    }

    @Bean
    MetricsDao metricsDao(TableClient tableClient) {
        var executor = threads.getExecutorService(config.getThreadPoolName(), "KikimrConfig.ThreadPoolName");
        YdbMetricsDao dao = new YdbMetricsDao(config, tableClient, executor);
        return DaoMetricsProxy.of(dao, MetricsDao.class, MetricRegistry.root());
    }

    @Bean
    public PingActorRunner removeOldRowsTask(MetricsDao dao) {
        Duration deletionPeriod = TimeConverter.protoToDuration(config.getDeletionPeriod());
        var executor = threads.getExecutorService(config.getThreadPoolName(), "KikimrConfig.ThreadPoolName");

        var task = PingActorRunner.newBuilder()
                .timer(threads.getSchedulerExecutorService())
                .executor(executor)
                .pingInterval(deletionPeriod)
                .backoffDelay(deletionPeriod)
                .operation("MetricsDao#removeOldRows")
                .onPing(value -> dao.removeOldRows())
                .build();
        task.schedule();
        return task;
    }

    @Bean
    public TableClient tableClient() {
        return ydbClients.getTableClient();
    }

    @Bean
    public SchemeClient schemeClient() {
        return ydbClients.getSchemeClient();
    }

    @Bean
    public MetricSupplier tableClientMetrics() {
        return ydbClients.getTableClientMetrics();
    }

    @Override
    public void close() {
        ydbClients.close();
    }
}
