package ru.yandex.market.graphouse.search.dao.ydb;

import java.util.List;
import java.util.Map;
import java.util.function.Consumer;

import javax.annotation.Nullable;

import com.yandex.ydb.table.description.TableDescription;
import com.yandex.ydb.table.result.ResultSetReader;
import com.yandex.ydb.table.values.ListType;
import com.yandex.ydb.table.values.ListValue;
import com.yandex.ydb.table.values.PrimitiveType;
import com.yandex.ydb.table.values.StructType;
import com.yandex.ydb.table.values.StructValue;
import com.yandex.ydb.table.values.TupleValue;

import ru.yandex.market.graphouse.search.MetricTreeStatus;
import ru.yandex.market.graphouse.search.dao.MetricArray;

import static com.yandex.ydb.table.values.PrimitiveValue.uint32;
import static com.yandex.ydb.table.values.PrimitiveValue.utf8;

/**
 * @author Vladimir Gordiychuk
 */
public class YdbLastUpdateTable {
    static final String TABLE_NAME = "LastUpdates";
    static final StructType METRIC_PK_TYPE = StructType.of(Map.of(
            "updateDateSeconds", PrimitiveType.uint32(),
            "name", PrimitiveType.utf8()
    ));
    static final ListType METRIC_PK_LIST_TYPE = ListType.of(METRIC_PK_TYPE);

    static TableDescription description() {
        return TableDescription.newBuilder()
                .addNullableColumn("updateDateSeconds", PrimitiveType.uint32())
                .addNullableColumn("name", PrimitiveType.utf8())
                .addNullableColumn("status", PrimitiveType.int32())
                .addNullableColumn("shardId", PrimitiveType.int32())
                .addNullableColumn("localId", PrimitiveType.int64())
                .setPrimaryKeys("updateDateSeconds", "name")
                .build();
    }

    static TupleValue key(int updateDateSeconds, String name) {
        return TupleValue.of(
                uint32(updateDateSeconds).makeOptional(),
                utf8(name).makeOptional());
    }

    static TupleValue key(int updateDateSeconds) {
        return TupleValue.of(uint32(updateDateSeconds).makeOptional());
    }

    private static StructValue keyToValue(Key key) {
        return METRIC_PK_TYPE.newValue(Map.of(
                "updateDateSeconds", uint32(key.updateDateSeconds),
                "name", utf8(key.name)
        ));
    }

    static ListValue keysToList(List<Key> keys) {
        var values = new StructValue[keys.size()];
        for (int index = 0; index < keys.size(); index++) {
            values[index] = keyToValue(keys.get(index));
        }

        return METRIC_PK_LIST_TYPE.newValueOwn(values);
    }

    static record Key(int updateDateSeconds, String name) {
    }

    static class ColumnReader {
        private final ResultSetReader resultSet;
        private final int updateDateSecondsIdx;
        private final int nameIdx;
        private final int statusIdx;
        private final int shardIdIdx;
        private final int localIdIdx;

        public ColumnReader(ResultSetReader resultSet) {
            this.resultSet = resultSet;
            this.updateDateSecondsIdx = resultSet.getColumnIndex("updateDateSeconds");
            this.nameIdx = resultSet.getColumnIndex("name");
            this.statusIdx = resultSet.getColumnIndex("status");
            this.shardIdIdx = resultSet.getColumnIndex("shardId");
            this.localIdIdx = resultSet.getColumnIndex("localId");
        }

        public boolean next() {
            return resultSet.next();
        }

        public Key readKey() {
            var updateDateSeconds = Math.toIntExact(resultSet.getColumn(updateDateSecondsIdx).getUint32());
            var name = resultSet.getColumn(nameIdx).getUtf8();
            return new Key(updateDateSeconds, name);
        }

        public long updateDataSeconds() {
            return resultSet.getColumn(updateDateSecondsIdx).getUint32();
        }

        public String name() {
            return resultSet.getColumn(nameIdx).getUtf8();
        }

        public MetricTreeStatus status() {
            return MetricTreeStatus.forId(resultSet.getColumn(statusIdx).getInt32());
        }

        public int shardId() {
            return resultSet.getColumn(shardIdIdx).getInt32();
        }

        public long localId() {
            return resultSet.getColumn(localIdIdx).getInt64();
        }
    }

    static class TableReader implements Consumer<ResultSetReader> {
        private final Consumer<MetricArray> consumer;
        private long lastUpdateDateSeconds;
        @Nullable
        private String lastName;
        private int readCount;

        public TableReader(Consumer<MetricArray> consumer) {
            this.consumer = consumer;
        }

        @Nullable
        public TupleValue getLastKey() {
            if (lastName == null) {
                return null;
            }

            return YdbLastUpdateTable.key(Math.toIntExact(lastUpdateDateSeconds), lastName);
        }

        public int getReadCount() {
            return readCount;
        }

        @Override
        public void accept(ResultSetReader resultSet) {
            int rows = resultSet.getRowCount();
            if (rows == 0) {
                return;
            }

            var reader = new ColumnReader(resultSet);
            var chunk = new MetricArray(rows);
            while (reader.next()) {
                chunk.add(reader.name(), reader.status(), reader.shardId(), reader.localId());
                lastUpdateDateSeconds = reader.updateDataSeconds();
            }

            consumer.accept(chunk);
            lastName = chunk.getName(rows - 1);
            readCount = rows;
        }
    }
}
