package ru.yandex.market.graphouse.search.dao.ydb;

import java.time.Duration;
import java.time.Instant;
import java.util.ArrayList;
import java.util.List;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.Executor;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.atomic.AtomicInteger;
import java.util.function.Consumer;
import java.util.function.Function;

import com.google.common.collect.Lists;
import com.google.common.util.concurrent.MoreExecutors;
import com.yandex.ydb.core.Result;
import com.yandex.ydb.core.Status;
import com.yandex.ydb.core.StatusCode;
import com.yandex.ydb.table.SessionRetryContext;
import com.yandex.ydb.table.TableClient;
import com.yandex.ydb.table.description.TableDescription;
import com.yandex.ydb.table.query.DataQueryResult;
import com.yandex.ydb.table.query.Params;
import com.yandex.ydb.table.result.ResultSetReader;
import com.yandex.ydb.table.settings.ExecuteDataQuerySettings;
import com.yandex.ydb.table.settings.ReadTableSettings;
import com.yandex.ydb.table.transaction.TxControl;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import ru.yandex.market.graphouse.search.dao.MetricArray;
import ru.yandex.market.graphouse.search.dao.MetricRow;
import ru.yandex.market.graphouse.search.dao.MetricsDao;
import ru.yandex.market.graphouse.search.dao.MetricsDaoMetrics;
import ru.yandex.solomon.config.TimeConverter;
import ru.yandex.solomon.config.protobuf.graphite.storage.TKikimrConfig;
import ru.yandex.solomon.util.actors.AsyncActorBody;
import ru.yandex.solomon.util.actors.AsyncActorRunner;
import ru.yandex.solomon.util.time.DurationUtils;

import static com.yandex.ydb.table.values.PrimitiveValue.uint32;
import static java.util.concurrent.CompletableFuture.completedFuture;
import static java.util.concurrent.CompletableFuture.failedFuture;

/**
 * This implementation uses 2 tables, the main table where we save metrics,
 * and the small one from which we can read last metrics very fast (aka cache).
 * Every insert should modify both tables in one transaction, so they are always synchronized.
 *
 *
 * @author Vladimir Gordiychuk
 */
public class YdbMetricsDao implements MetricsDao {
    private static final Logger logger = LoggerFactory.getLogger(YdbMetricsDao.class);

    private final Duration ttl;
    private final int maxBatchSize;
    private final int maxInFlight;
    private final Executor executor;
    private final YdbQueries queries;
    private final MetricsDaoMetrics metrics;
    private final SessionRetryContext retryCtx;
    private final TableClient tableClient;

    public YdbMetricsDao(TKikimrConfig config, TableClient tableClient, Executor executor) {
        this.ttl = TimeConverter.protoToDuration(config.getLastUpdateTtl());
        this.maxBatchSize = config.getBatchSize();
        this.maxInFlight = config.getMaxSavingInFlight();
        this.executor = executor;
        this.queries = new YdbQueries(config.getSchemaRoot());
        this.metrics = new MetricsDaoMetrics();
        this.tableClient = tableClient;
        this.retryCtx = SessionRetryContext.create(tableClient)
                .maxRetries(10)
                .sessionSupplyTimeout(Duration.ofSeconds(30))
                .build();
    }

    public CompletableFuture<Void> createSchemaForTests() {
        return CompletableFuture.allOf(
                createTable(queries.metricsTable, YdbMetricsTable.description()),
                createTable(queries.metricsLastUpdatesTable, YdbLastUpdateTable.description()));
    }

    @Override
    public CompletableFuture<Void> saveMetrics(List<MetricRow> metrics) {
        return splitRun(metrics, chunk -> {
            this.metrics.addMetrics.add(chunk.size());
            Params params = Params.of("$rows", YdbMetricsTable.metricsToList(chunk));
            return execute(queries.saveQuery, params)
                    .thenAccept(r -> r.expect("cannot save metrics"));
        });
    }

    @Override
    public CompletableFuture<Integer> loadAllMetrics(Consumer<MetricArray> consumer) {
        var it = YdbMetricsTable.PrimaryKeyRange.range().split(1024).iterator();
        AtomicInteger rows = new AtomicInteger();
        AsyncActorBody body = () -> {
            if (!it.hasNext()) {
                return completedFuture(AsyncActorBody.DONE_MARKER);
            }

            return loadAllMetrics(consumer, it.next()).thenAccept(rows::addAndGet);
        };

        AsyncActorRunner runner = new AsyncActorRunner(body, executor, 16);
        return runner.start().thenApply(ignore -> rows.get());
    }

    private CompletableFuture<Integer> loadAllMetrics(Consumer<MetricArray> consumer, YdbMetricsTable.PrimaryKeyRange range) {
        var doneFuture = new LoadCompletableFuture();
        var reader = new YdbMetricsTable.TableReader(metricDbRows -> {
            if (doneFuture.isDone()) {
                throw new IllegalStateException("not actual anymore");
            }

            metrics.readMetrics.add(metricDbRows.size());
            doneFuture.addOperation();
            executor.execute(() -> {
                try {
                    consumer.accept(metricDbRows);
                    doneFuture.finishOperation();
                } catch (Throwable e) {
                    doneFuture.completeExceptionally(e);
                }
            });
        });

        var retryCtx = SessionRetryContext.create(tableClient)
                .maxRetries(Integer.MAX_VALUE)
                .backoffSlot(Duration.ofSeconds(1))
                .backoffCeiling(10)
                .sessionSupplyTimeout(Duration.ofSeconds(30))
                .build();

        long startNanos = System.nanoTime();
        retryCtx.supplyStatus(session -> {
            var settings = ReadTableSettings.newBuilder()
                    .timeout(5, TimeUnit.MINUTES)
                    .columns("name", "status", "shardId", "localId")
                    .orderedRead(true);

            var from = range.beginInclusive();
            if (from != null) {
                settings.fromKeyInclusive(from);
            }

            var to = range.endExclusive();
            if (to != null) {
                settings.toKeyExclusive(to);
            }

            var lastKey = reader.getLastKey();
            if (lastKey != null) {
                settings.fromKeyExclusive(lastKey);
            }

            return session.readTable(queries.metricsTable, settings.build(), reader)
                    .thenApply(status -> {
                        switch (status.getCode()) {
                            case CLIENT_DEADLINE_EXCEEDED:
                            case TIMEOUT:
                                return Status.of(StatusCode.UNAVAILABLE, status.getIssues());
                        }

                        if (!status.isSuccess()) {
                            logger.warn("{} failed load, status {}", range, status);
                        }

                        return status;
                    });
        }).thenAccept(status -> {
            status.expect("can't read table " + queries.metricsTable);
            var tookMs = TimeUnit.NANOSECONDS.toMillis(System.nanoTime() - startNanos);
            logger.info("{} loaded {} rows, took {} ", range, reader.getReadCount(), DurationUtils.formatDurationMillis(tookMs));
            doneFuture.finishOperation(reader.getReadCount());
        }).exceptionally(e -> {
            logger.error("{} failed load", range, e);
            doneFuture.completeExceptionally(e);
            return null;
        });
        return doneFuture;
    }

    @Override
    public CompletableFuture<Integer> loadNewMetrics(Consumer<MetricArray> consumer, int startTimestampSeconds) {
        return execute(queries.selectNewQuery, Params.of("$startInc", uint32(startTimestampSeconds)))
                .thenCompose(result -> {
                    DataQueryResult dataResult = result.expect("cannot select fresh metrics");
                    ResultSetReader resultSet = dataResult.getResultSet(0);
                    if (resultSet.isTruncated() || resultSet.getRowCount() >= 1000) {
                        return loadNewMetricsReadTable(consumer, startTimestampSeconds);
                    }

                    var reader = new YdbLastUpdateTable.TableReader(metricDbRows -> {
                        metrics.readFreshMetrics.add(metricDbRows.size());
                        consumer.accept(metricDbRows);
                    });
                    reader.accept(resultSet);
                    return completedFuture(reader.getReadCount());
                });
    }

    public CompletableFuture<Integer> loadNewMetricsReadTable(Consumer<MetricArray> consumer, int startTimestampSeconds) {
        var reader = new YdbLastUpdateTable.TableReader(metricDbRows -> {
            metrics.readFreshMetrics.add(metricDbRows.size());
            consumer.accept(metricDbRows);
        });
        return retryCtx.supplyStatus(session -> {
            var settings = ReadTableSettings.newBuilder()
                    .timeout(30, TimeUnit.MINUTES)
                    .orderedRead(true);

            var lastKey = reader.getLastKey();
            if (lastKey != null) {
                settings.fromKeyExclusive(lastKey);
            } else {
                settings.fromKeyInclusive(YdbLastUpdateTable.key(startTimestampSeconds));
            }

            return session.readTable(queries.metricsLastUpdatesTable, settings.build(), reader);
        }).thenApply(status -> {
            status.expect("can't read table " + queries.metricsLastUpdatesTable);
            return reader.getReadCount();
        });
    }

    @Override
    public CompletableFuture<Void> remove(List<String> names) {
        return splitRun(names, chunk -> {
            Params params = Params.of("$keys", YdbMetricsTable.keysToList(names));
            return execute(queries.deleteQuery, params)
                    .thenAccept(result -> result.expect("cannot delete metrics"));
        });
    }

    @Override
    public CompletableFuture<Void> removeOldRows() {
        int deleteRowsOlderThan = Math.toIntExact(Instant.now().minus(ttl).getEpochSecond());
        logger.info("delete rows where updateDateSeconds < {}", deleteRowsOlderThan);

        AsyncActorBody body = () -> {
            return execute(queries.ttlSelectQuery, Params.of("$deletionTime", uint32(deleteRowsOlderThan)))
                    .thenCompose(result -> {
                        DataQueryResult dataResult = result.expect("cannot select metrics to delete");
                        ResultSetReader resultSet = dataResult.getResultSet(0);
                        if (!resultSet.isTruncated() && resultSet.getRowCount() == 0) {
                            return completedFuture(AsyncActorBody.DONE_MARKER);
                        }

                        var keys = new ArrayList<YdbLastUpdateTable.Key>(resultSet.getRowCount());
                        var reader = new YdbLastUpdateTable.ColumnReader(resultSet);
                        while (reader.next()) {
                            keys.add(reader.readKey());
                        }

                        logger.info("Success to select rows for deletion, size:{}", keys.size());
                        metrics.deleteRowsCounter.add(keys.size());
                        metrics.deleteFreshMetrics.add(keys.size());
                        long startDeletionNanos = System.nanoTime();
                        return execute(queries.ttlDeleteQuery, Params.of("$keys", YdbLastUpdateTable.keysToList(keys)))
                                .thenApply(deleteResult -> {
                                    deleteResult.expect("cannot delete rows from fresh table");
                                    return null;
                                })
                                .whenComplete((ignore, e) -> {
                                    if (e != null) {
                                        logger.error("Failed to delete rows from:{}", queries.metricsLastUpdatesTable, e);
                                        metrics.failDeletionCounter.inc();
                                    } else {
                                        logger.info("Success to delete rows from:{}", queries.metricsLastUpdatesTable);
                                        metrics.successDeletionCounter.inc();
                                    }
                                    metrics.histogramDeletionTime.record(TimeUnit.NANOSECONDS.toMillis(System.nanoTime() - startDeletionNanos));
                                });
                    });
        };

        AsyncActorRunner runner = new AsyncActorRunner(body, executor, 1);
        return runner.start();
    }

    private CompletableFuture<Void> createTable(String name, TableDescription description) {
        return retryCtx.supplyStatus(s -> s.createTable(name, description))
                .thenAccept(status -> status.expect("cannot create table " + name));
    }

    private CompletableFuture<Result<DataQueryResult>> execute(String query, Params params) {
        try {
            return retryCtx.supplyResult(s -> {
                var settings = new ExecuteDataQuerySettings().keepInQueryCache();
                var tx = TxControl.serializableRw();
                return s.executeDataQuery(query, tx, params, settings);
            });
        } catch (Throwable t) {
            return failedFuture(t);
        }
    }

    private <T> CompletableFuture<Void> splitRun(List<T> metrics, Function<List<T>, CompletableFuture<Void>> fn) {
        if (metrics.size() < maxBatchSize) {
            return fn.apply(metrics);
        }

        var it = Lists.partition(metrics, maxBatchSize).iterator();
        AsyncActorBody body = () -> {
            if (!it.hasNext()) {
                return completedFuture(AsyncActorBody.DONE_MARKER);
            }
            return fn.apply(it.next());
        };

        AsyncActorRunner actorRunner = new AsyncActorRunner(body, MoreExecutors.directExecutor(), maxInFlight);
        return actorRunner.start();
    }

}
