package ru.yandex.market.graphouse.search.dao.ydb;

import java.util.ArrayList;
import java.util.List;
import java.util.Map;
import java.util.function.Consumer;

import javax.annotation.Nullable;

import com.yandex.ydb.table.description.TableDescription;
import com.yandex.ydb.table.result.ResultSetReader;
import com.yandex.ydb.table.values.ListType;
import com.yandex.ydb.table.values.ListValue;
import com.yandex.ydb.table.values.PrimitiveType;
import com.yandex.ydb.table.values.StructType;
import com.yandex.ydb.table.values.StructValue;
import com.yandex.ydb.table.values.TupleValue;

import ru.yandex.market.graphouse.search.MetricTreeStatus;
import ru.yandex.market.graphouse.search.dao.MetricArray;
import ru.yandex.market.graphouse.search.dao.MetricRow;

import static com.yandex.ydb.table.values.PrimitiveValue.int32;
import static com.yandex.ydb.table.values.PrimitiveValue.int64;
import static com.yandex.ydb.table.values.PrimitiveValue.uint32;
import static com.yandex.ydb.table.values.PrimitiveValue.utf8;

/**
 * @author Vladimir Gordiychuk
 */
public class YdbMetricsTable {
    static final String TABLE_NAME = "Metrics";
    static final StructType METRIC_TYPE = StructType.of(Map.of(
            "hash", PrimitiveType.uint32(),
            "name", PrimitiveType.utf8(),
            "status", PrimitiveType.int32(),
            "updateDateSeconds", PrimitiveType.uint32(),
            "shardId", PrimitiveType.int32(),
            "localId", PrimitiveType.int64()
    ));
    static final ListType METRIC_LIST_TYPE = ListType.of(METRIC_TYPE);

    static final StructType METRIC_PK_TYPE = StructType.of(Map.of(
            "hash", PrimitiveType.uint32(),
            "name", PrimitiveType.utf8()
    ));
    static final ListType METRIC_PK_LIST_TYPE = ListType.of(METRIC_PK_TYPE);

    static TableDescription description() {
        return TableDescription.newBuilder()
                .addNullableColumn("hash", PrimitiveType.uint32())
                .addNullableColumn("name", PrimitiveType.utf8())
                .addNullableColumn("status", PrimitiveType.int32())
                .addNullableColumn("updateDateSeconds", PrimitiveType.uint32())
                .addNullableColumn("shardId", PrimitiveType.int32())
                .addNullableColumn("localId", PrimitiveType.int64())
                .setPrimaryKeys("hash", "name")
                .build();
    }

    static TupleValue key(String name) {
        return TupleValue.of(
                uint32(name.hashCode()).makeOptional(),
                utf8(name).makeOptional());
    }

    private static StructValue metricToValue(MetricRow metric) {
        return METRIC_TYPE.newValue(Map.of(
                "hash", uint32(metric.name().hashCode()),
                "name", utf8(metric.name()),
                "status", int32(metric.status()),
                "updateDateSeconds", uint32(metric.updateDateSeconds()),
                "shardId", int32(metric.shardId()),
                "localId", int64(metric.localId())
        ));
    }

    static ListValue metricsToList(List<MetricRow> metrics) {
        var values = new StructValue[metrics.size()];
        for (int index = 0; index < metrics.size(); index++) {
            values[index] = metricToValue(metrics.get(index));
        }
        return METRIC_LIST_TYPE.newValueOwn(values);
    }

    private static StructValue keyToValue(String name) {
        return METRIC_PK_TYPE.newValue(Map.of(
                "hash", uint32(name.hashCode()),
                "name", utf8(name)
        ));
    }

    static ListValue keysToList(List<String> keys) {
        var values = new StructValue[keys.size()];
        for (int index = 0; index < keys.size(); index++) {
            values[index] = keyToValue(keys.get(index));
        }

        return METRIC_PK_LIST_TYPE.newValueOwn(values);
    }

    static class ColumnReader {
        private final ResultSetReader resultSet;
        private final int nameIdx;
        private final int statusIdx;
        private final int shardIdIdx;
        private final int localIdIdx;

        public ColumnReader(ResultSetReader resultSet) {
            this.resultSet = resultSet;
            this.nameIdx = resultSet.getColumnIndex("name");
            this.statusIdx = resultSet.getColumnIndex("status");
            this.shardIdIdx = resultSet.getColumnIndex("shardId");
            this.localIdIdx = resultSet.getColumnIndex("localId");
        }

        public boolean next() {
            return resultSet.next();
        }

        public String name() {
            return resultSet.getColumn(nameIdx).getUtf8();
        }

        public MetricTreeStatus status() {
            return MetricTreeStatus.forId(resultSet.getColumn(statusIdx).getInt32());
        }

        public int shardId() {
            return resultSet.getColumn(shardIdIdx).getInt32();
        }

        public long localId() {
            return resultSet.getColumn(localIdIdx).getInt64();
        }
    }

    static class TableReader implements Consumer<ResultSetReader> {
        private final Consumer<MetricArray> consumer;
        @Nullable
        private String last;
        private int readCount;

        public TableReader(Consumer<MetricArray> consumer) {
            this.consumer = consumer;
        }

        @Nullable
        public TupleValue getLastKey() {
            if (last == null) {
                return null;
            }

            return YdbMetricsTable.key(last);
        }

        public int getReadCount() {
            return readCount;
        }

        @Override
        public void accept(ResultSetReader rs) {
            int rows = rs.getRowCount();
            if (rows == 0) {
                return;
            }

            var reader = new ColumnReader(rs);
            var chunk = new MetricArray(rows);
            while (rs.next()) {
                chunk.add(reader.name(), reader.status(), reader.shardId(), reader.localId());
            }

            consumer.accept(chunk);
            last = chunk.getName(rows - 1);
            readCount = rows;
        }
    }

    static record PrimaryKeyRange(long beginInc, long endExc) {
        private static final long MIN_VALUE = 0;
        private static final long MAX_VALUE = Integer.toUnsignedLong(Integer.MAX_VALUE) + 1;

        public PrimaryKeyRange {
            if (beginInc >= endExc) {
                throw new IllegalArgumentException("begin " + beginInc + " >= end " + endExc);
            }

            if (beginInc < MIN_VALUE) {
                throw new IllegalArgumentException("begin " + beginInc + " < " + MIN_VALUE);
            }

            if (endExc > MAX_VALUE) {
                throw new IllegalArgumentException("end > " + MAX_VALUE);
            }
        }

        public static PrimaryKeyRange range() {
            return new PrimaryKeyRange(MIN_VALUE, MAX_VALUE);
        }

        public List<PrimaryKeyRange> split(int part) {
            long size = (long) Math.ceil((double)(endExc - beginInc) / (double) part);
            List<PrimaryKeyRange> ranges = new ArrayList<>(part);
            long begin = beginInc;
            while (begin < endExc) {
                long end = Math.min(endExc, begin + size);
                ranges.add(new PrimaryKeyRange(begin, end));
                begin = end;
            }
            return ranges;
        }

        @Nullable
        public TupleValue beginInclusive() {
            if (beginInc == MIN_VALUE) {
                return null;
            }

            return TupleValue.of(uint32((int) beginInc).makeOptional());
        }

        @Nullable
        public TupleValue endExclusive() {
            if (endExc == MAX_VALUE) {
                return null;
            }

            return TupleValue.of(uint32((int) endExc).makeOptional());
        }
    }

}
