package ru.yandex.market.graphouse.search.tree;

import java.util.List;
import java.util.concurrent.locks.ReentrantLock;

import javax.annotation.Nonnull;
import javax.annotation.Nullable;
import javax.annotation.ParametersAreNonnullByDefault;

import com.google.common.collect.Interner;
import com.google.common.collect.Interners;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import ru.yandex.market.graphouse.retention.RetentionManager;
import ru.yandex.market.graphouse.search.MetricTree;
import ru.yandex.market.graphouse.stockpile.GraphouseStockpileIdGenerator;
import ru.yandex.stockpile.client.shard.StockpileMetricId;

@ParametersAreNonnullByDefault
public class Dir extends MetricBase {
    private static final Logger log = LoggerFactory.getLogger(Dir.class);
    private static final Interner<String> metricInterner = Interners.newStrongInterner();

    private volatile Object metrics = null;
    private volatile Object dirs = null;

    public Dir(String name) {
        super(name);

        if (name.isEmpty()) {
            throw new IllegalArgumentException();
        }
    }

    /** Root */
    public Dir() {
        super("");
    }

    @Nonnull
    public Dir getOrCreateDir(MetricTree tree, String name) {
        Dir oldDir = getDir(name);
        if (oldDir != null) {
            return oldDir;
        }
        name = metricInterner.intern(name);
        Dir newDir = new Dir(name);
        oldDir = putDirIfAbsent(newDir, getLock());
        if (oldDir == null) {
            tree.registerNewDir();
            return newDir;
        } else {
            return oldDir;
        }
    }

    @Nonnull
    public MetricName getOrCreateMetric(
        MetricTree tree,
        String name,
        String fullName,
        GraphouseStockpileIdGenerator idSource,
        RetentionManager mgr
    ) {
        MetricName oldMetric = getMetric(name);

        if (oldMetric == null) {
            return saveNewMetric(tree, name, fullName, idSource.generateMetricId(), mgr, idSource.isStockpileIdFromDatabase());
        } else {
            if (idSource.isStockpileIdFromDatabase()) {
                StockpileMetricId newId = idSource.generateMetricId();
                if (oldMetric.isFromDb()) {
                    if (newId.equals(oldMetric.getStockpileId())) {
                        return oldMetric;
                    } else {
                        // Существующая метрика не соответствует эталонному состоянию в БД, её надо пересоздать
                        log.error("Metric " + fullName + " was marked as loaded from db," +
                            " but had stockpile id inconsistent with db:" +
                            " Old id: " + oldMetric.getStockpileId() +
                            " New id: " + newId
                        );
                    }
                }
                oldMetric.setStockpileIdFromDb(newId);
            }
            return oldMetric;
        }
    }

    @Nonnull
    private MetricName saveNewMetric(MetricTree tree, String name, String fullName, StockpileMetricId id, RetentionManager mgr, boolean isFromDb) {
        name = metricInterner.intern(name);
        byte retentionId = mgr.getRetentionIdForMetric(fullName);
        MetricName newMetricName = new MetricName(name, id, isFromDb, retentionId);
        MetricName oldMetricName = putMetricIfAbsent(newMetricName, getLock());
        if (oldMetricName == null) {
            tree.registerNewMetric();
            return newMetricName;
        } else {
            return oldMetricName;
        }
    }

    public int sizeDirs() {
        Object dirs = this.dirs;
        if (dirs == null) {
            return 0;
        } else if (dirs instanceof Dir) {
            return 1;
        } else {
            var map = (ConcurrentExpandingMetricBaseMap<Dir>) dirs;
            return map.getSize();
        }
    }

    @Nonnull
    public Iterable<Dir> listDirs() {
        Object dirs = this.dirs;
        if (dirs == null) {
            return List.of();
        } else if (dirs instanceof Dir) {
            return List.of((Dir) dirs);
        } else {
            var map = (ConcurrentExpandingMetricBaseMap<Dir>) dirs;
            return map.values();
        }
    }

    public int sizeMetrics() {
        Object metrics = this.metrics;
        if (metrics == null) {
            return 0;
        } else if (metrics instanceof MetricName) {
            return 1;
        } else {
            var map = (ConcurrentExpandingMetricBaseMap<MetricName>) metrics;
            return map.getSize();
        }
    }

    public Iterable<MetricName> listMetrics() {
        Object metrics = this.metrics;
        if (metrics == null) {
            return List.of();
        } else if (metrics instanceof MetricName) {
            return List.of((MetricName) metrics);
        } else {
            var map = (ConcurrentExpandingMetricBaseMap<MetricName>) metrics;
            return map.values();
        }
    }

    @Nullable
    public Dir getDir(String dirName) {
        Object dirs = this.dirs;
        if (dirs == null) {
            return null;
        } else if (dirs instanceof Dir) {
            var dir = (Dir) dirs;
            if (dir.getLastNameChunk().equals(dirName)) {
                return dir;
            } else {
                return null;
            }
        } else {
            var map = (ConcurrentExpandingMetricBaseMap<Dir>) dirs;
            return map.get(dirName);
        }
    }

    @Nullable
    public MetricName getMetric(String metricName) {
        Object metrics = this.metrics;
        if (metrics == null) {
            return null;
        } else if (metrics instanceof MetricName) {
            var metric = (MetricName) metrics;
            if (metric.name.equals(metricName)) {
                return metric;
            } else {
                return null;
            }
        } else {
            var map = (ConcurrentExpandingMetricBaseMap<MetricName>) metrics;
            return map.get(metricName);
        }
    }

    @Nullable
    public MetricName putMetricIfAbsent(MetricName newMetric, ReentrantLock lock) {
        lock.lock();
        try {
            Object metrics = this.metrics;
            if (metrics == null) {
                this.metrics = newMetric;
                return null;
            } else if (metrics instanceof MetricName) {
                var metric = (MetricName) metrics;
                if (metric.name.equals(newMetric.name)) {
                    return metric;
                } else {
                    var map = new ConcurrentExpandingMetricBaseMap<MetricName>();
                    map.putIfAbsent(metric, lock);
                    map.putIfAbsent(newMetric, lock);
                    this.metrics = map;
                    return null;
                }
            } else {
                var map = (ConcurrentExpandingMetricBaseMap<MetricName>) metrics;
                return map.putIfAbsent(newMetric, lock);
            }
        } finally {
            lock.unlock();
        }
    }

    @Nullable
    public Dir putDirIfAbsent(Dir newDir, ReentrantLock lock) {
        lock.lock();
        try {
            Object dirs = this.dirs;
            if (dirs == null) {
                this.dirs = newDir;
                return null;
            } else if (dirs instanceof Dir) {
                var dir = (Dir) dirs;
                if (dir.getLastNameChunk().equals(newDir.getLastNameChunk())) {
                    return dir;
                } else {
                    ConcurrentExpandingMetricBaseMap<Dir> map = new ConcurrentExpandingMetricBaseMap<>();
                    map.putIfAbsent(dir, lock);
                    map.putIfAbsent(newDir, lock);
                    this.dirs = map;
                    return null;
                }
            } else {
                var map = (ConcurrentExpandingMetricBaseMap<Dir>) dirs;
                return map.putIfAbsent(newDir, lock);
            }
        } finally {
            lock.unlock();
        }
    }

    public void updateStats(TreeStats stats) {
        if (metrics != null) {
            int metricCount = sizeMetrics();
            int oldStat = stats.dirCountByMetricCount.getOrDefault(metricCount, 0);
            stats.dirCountByMetricCount.put(metricCount, oldStat + 1);
        }
        if (dirs != null) {
            int dirsCount = sizeDirs();
            int oldStat = stats.dirCountBySubdirCount.getOrDefault(dirsCount, 0);
            stats.dirCountBySubdirCount.put(dirsCount, oldStat + 1);
            for (var dir : listDirs()) {
                dir.updateStats(stats);
            }
            if (metrics == null) {
                stats.onlyDirs++;
            } else {
                stats.both++;
            }
        } else {
            if (metrics != null) {
                stats.onlyMetrics++;
            } else {
                stats.none++;
            }
        }
    }
}
