package ru.yandex.market.graphouse.search.tree;

import java.util.List;

import javax.annotation.Nonnull;
import javax.annotation.ParametersAreNonnullByDefault;

import ru.yandex.market.graphouse.search.MetricDescription;
import ru.yandex.market.graphouse.search.MetricTreeStatus;
import ru.yandex.stockpile.client.shard.StockpileMetricId;

/**
 * @author Stepan Koltsov
 */
@ParametersAreNonnullByDefault
public class MetricDescriptionImpl<D extends MetricBase> implements MetricDescription {

    private final List<Dir> parents;
    @Nonnull
    public final D self;

    public MetricDescriptionImpl(List<Dir> parents, D self) {
        this.parents = parents;
        this.self = self;
    }

    private boolean isRoot() {
        return parents.isEmpty();
    }

    @Override
    public String getName() {
        if (isRoot()) {
            return "ROOT";
        }
        StringBuilder sb = new StringBuilder();
        for (int index = 1; index < parents.size(); index++) {
            sb.append(parents.get(index).name);
            sb.append('.');
        }
        sb.append(self.name);
        return sb.toString();
    }

    @Override
    public MetricTreeStatus getStatus() {
        return self.getStatus();
    }

    @Override
    public boolean isDir() {
        return self instanceof Dir;
    }

    @Override
    public long getUpdateTimeMillis() {
        return self.getUpdateTimeMillis();
    }

    @Override
    public StockpileMetricId getStockpileId() {
        return ((MetricName) self).getStockpileId();
    }

    @Override
    public short getRetentionId() {
        return ((MetricName) self).getRetentionId();
    }

    @Override
    public int getLastWriteTimestamp() {
        if (self instanceof MetricName) {
            var metric = (MetricName) self;
            return metric.getLastWriteTimestampSeconds();
        }
        return 0;
    }

    @Override
    public void updateLastWriteTimestamp(int timestamp) {
        if (self instanceof MetricName) {
            var metric = (MetricName) self;
            metric.setLastWriteTimestampSeconds(timestamp);
        }
    }

    @Override
    public MetricBaseZip<? extends MetricBase> asZip() {
        if (isDir()) {
            return new DirZip(DirZip.of(parents), (Dir) self);
        }

        return new MetricNameZip(DirZip.of(parents), (MetricName) self);
    }

    @Override
    public String toString() {
        return "MetricBaseZip{name=" + getName() + '}';
    }
}
