package ru.yandex.market.graphouse.search.util;

import java.util.Collections;
import java.util.Iterator;
import java.util.NoSuchElementException;
import java.util.concurrent.locks.ReentrantLock;
import java.util.stream.Stream;
import java.util.stream.StreamSupport;

import javax.annotation.Nonnull;
import javax.annotation.Nullable;
import javax.annotation.ParametersAreNonnullByDefault;

import ru.yandex.misc.algo.ht.Accessor;
import ru.yandex.misc.algo.ht.LinearProbeOpenHashTableImpl;
import ru.yandex.misc.algo.ht.ResizeAccessor;
import ru.yandex.misc.jvm.UnsafeUtils;

/**
 * @author Maksim Leonov (nohttp@)
 *
 * Memory-effective collection with a map-like interface, which allows adding and retrieving non-null elements concurrently.
 */
@ParametersAreNonnullByDefault
abstract class ConcurrentExpandingMapBase<K, V, T> {
    @Nullable
    private volatile T table = null;
    private int size = 0;

    public abstract K keyForValue(V value);

    public abstract int hash(K key);

    /* Table access utils */

    protected abstract T allocateTable(int size);

    @Nullable
    protected abstract V getFromTable(T table, int idx);

    protected abstract int tableLength(T table);

    protected abstract void setInTable(T table, int idx, V value);

    protected void copyFromTable(T fromTable, int fromIdx, T toTable, int toIdx) {
        setInTable(toTable, toIdx, getFromTable(fromTable, fromIdx));
    }

    /* main logic */

    @Nullable
    public V get(K key) {
        return getImpl(key, table);
    }

    @Nullable
    public V putIfAbsent(V value, ReentrantLock lock) {
        K key = keyForValue(value);
        lock.lock();
        try {
            if (table == null) {
                table = allocateTable(1);
            }
            UnsafeUtils.UNSAFE.storeFence();
            return putIfAbsentWithLockImpl(key, value, table);
        } finally {
            lock.unlock();
        }
    }

    @Nonnull
    public Iterable<V> values() {
        return valuesImpl(table);
    }

    @Nonnull
    public Stream<V> valuesStream() {
        return StreamSupport.stream(values().spliterator(), false);
    }

    @Nullable
    private V getImpl(K key, @Nullable T tableSnapshot) {
        if (tableSnapshot == null) {
            return null;
        }
        UnsafeUtils.UNSAFE.loadFence();
        int keyHashCode = hash(key);
        int index = LinearProbeOpenHashTableImpl.get(
            new AccessorImpl(tableSnapshot),
            keyHashCode,
            i -> checkKeyAtIndex(key, keyHashCode, i, tableSnapshot));
        if (index >= 0) {
            return getFromTable(tableSnapshot, index);
        } else {
            return null;
        }
    }

    @Nullable
    private V putIfAbsentWithLockImpl(K key, V value, T tableSnapshot) {
        int keyHashCode = hash(key);
        AccessorImpl accessor = new AccessorImpl(tableSnapshot);

        tableSnapshot = null; // Now we need to use accessor's field since it can be overwritten with an expanded version

        int index = LinearProbeOpenHashTableImpl.findPositionForPut(
            accessor,
            keyHashCode,
            i -> checkKeyAtIndex(key, keyHashCode, i, accessor.tableSnapshot),
            new ResizeAccessorImpl(accessor),
            true);

        if (index >= 0) {
            return getFromTable(accessor.tableSnapshot, index);
        } else {
            index = index & 0x7fffffff;
            setInTable(accessor.tableSnapshot, index, value);
            size++;
            return null;
        }
    }

    @Nonnull
    private Iterable<V> valuesImpl(@Nullable T tableSnapshot) {
        if (tableSnapshot == null) {
            return Collections.emptyList();
        }
        return () -> new Iterator<V>() {
            int nextIdx = findNext(-1);

            @Override
            public boolean hasNext() {
                return nextIdx != -1;
            }

            @Override
            public V next() {
                if (nextIdx == -1) {
                    throw new NoSuchElementException();
                }
                int idx = nextIdx;
                nextIdx = findNext(idx);
                return getFromTable(tableSnapshot, idx);
            }

            private int findNext(int current) {
                for (int i = current + 1; i < tableLength(tableSnapshot); i++) {
                    if (getFromTable(tableSnapshot, i) != null) {
                        return i;
                    }
                }
                return -1;
            }
        };
    }

    private int hashCodeAt(int index, T table) {
        return hash(keyForValue(getFromTable(table, index)));
    }

    private boolean checkKeyAtIndex(K in, int keyHashCode, int idx, T tableSnapshot) {
        K keyAtI = keyForValue(getFromTable(tableSnapshot, idx));
        return hash(keyAtI) == keyHashCode && keyAtI.equals(in);
    }

    public int getSize() {
        return size;
    }

    private class AccessorImpl implements Accessor {
        private T tableSnapshot;

        AccessorImpl(T tableSnapshot) {
            this.tableSnapshot = tableSnapshot;
        }

        @Override
        public int tableSize() {
            return tableLength(tableSnapshot);
        }

        @Override
        public int dataSize() {
            return size;
        }

        @Override
        public boolean isUsedAt(int index) {
            return getFromTable(this.tableSnapshot, index) != null;
        }

        @Override
        public int hashCodeAt(int index) {
            return ConcurrentExpandingMapBase.this.hashCodeAt(index, tableSnapshot);
        }

    }

    private class ResizeAccessorImpl implements ResizeAccessor {
        private final AccessorImpl accessor;
        private T newTable;

        ResizeAccessorImpl(AccessorImpl accessor) {
            this.accessor = accessor;
        }

        @Override
        public void allocate(int size) {
            this.newTable = ConcurrentExpandingMapBase.this.allocateTable(size);
        }

        @Override
        public boolean isUsedAt(int index) {
            return getFromTable(newTable, index) != null;
        }

        @Override
        public void moveFromOriginal(int from, int to) {
            copyFromTable(accessor.tableSnapshot, from, this.newTable, to);
        }

        @Override
        public void replaceDefault() {
            accessor.tableSnapshot = newTable;
            ConcurrentExpandingMapBase.this.table = newTable;
            newTable = null;
        }
    }
}
