package ru.yandex.market.graphouse.server;

import java.util.Arrays;
import java.util.Collections;
import java.util.List;
import java.util.concurrent.TimeUnit;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Component;

import ru.yandex.market.graphouse.retention.RetentionManager;
import ru.yandex.market.graphouse.search.MetricAddException;
import ru.yandex.market.graphouse.search.MetricResponseStatus;
import ru.yandex.market.graphouse.search.MetricSearch;
import ru.yandex.market.graphouse.search.tree.MetricNameZip;
import ru.yandex.solomon.util.time.InstantUtils;
import ru.yandex.stockpile.client.shard.StockpileMetricId;

/**
 * @author Dmitry Andreev <a href="mailto:AndreevDm@yandex-team.ru"/>
 * @date 08/05/15
 */
@Component
public class MetricFactory {
    private static final Logger log = LoggerFactory.getLogger(MetricFactory.class);

    @Autowired
    private MetricSearch metricSearch;
    @Autowired
    private MetricValidator metricValidator;
    @Autowired
    private RetentionManager retentionManager;

    private boolean redirectHostMetrics = false;
    private String hostMetricDir = "HOST";

    private static final List<String> hostPostfixes = Collections.unmodifiableList(Arrays.asList("yandex_net", "yandex_ru"));

    /**
     * Works like String.split(' ', 3) but without Array allocations
     * @param line
     * @return
     * @throws MetricAddException
     */
    private static String[] splitLine(String line) throws MetricAddException {
        int sep1idx = line.indexOf(' ');
        if (sep1idx < 0) {
            throw MetricAddException.problem(MetricResponseStatus.REJECTED_MALFORMED);
        }
        int sep2idx = line.indexOf(' ', sep1idx + 1);
        if (sep2idx < 0) {
            throw MetricAddException.problem(MetricResponseStatus.REJECTED_MALFORMED);
        }
        return new String[]{
            line.substring(0, sep1idx),
            line.substring(sep1idx + 1, sep2idx),
            line.substring(sep2idx + 1, line.length())
        };
    }

    /**
     * Валидирует метрику и в случае успеха создаёт или обновляет текущую.
     *
     * @param line    через пробел название метрики, значение, метка времени
     * @return созданную или обновленную метрику,
     */
    public AnyMetric createMetric(String line, String ipAddress) throws MetricAddException {
        String[] splits = splitLine(line);
        String name = splits[0];
        MetricResponseStatus validatorStatus = metricValidator.validate(name, ipAddress);
        if (validatorStatus != MetricResponseStatus.OK) {
            throw MetricAddException.problem(validatorStatus);
        }
        try {
            double value = Double.parseDouble(splits[1]);
            if (!Double.isFinite(value)) {
                throw MetricAddException.problem(MetricResponseStatus.REJECTED_MALFORMED);
            }
            int timeSeconds = Integer.parseInt(splits[2]);
            if (!InstantUtils.isGoodSeconds(timeSeconds)) {
                throw MetricAddException.problem(MetricResponseStatus.REJECTED_MALFORMED);
            }
            long timeMillis = TimeUnit.SECONDS.toMillis(timeSeconds);
            name = processName(name);

            MetricNameZip metricDescription = (MetricNameZip) metricSearch.add(name);
            metricDescription.updateLastWriteTimestamp((int) (TimeUnit.MILLISECONDS.toSeconds(System.currentTimeMillis())));

            short stockpilePolicyId = retentionManager.getStockpilePolicyId(metricDescription);
            if (!metricDescription.self.isFromDb()) {
                return new UnresolvedMetric(metricDescription.self, timeMillis, value, stockpilePolicyId);
            }

            StockpileMetricId id = metricDescription.getStockpileId();
            return new ResolvedMetric(id.getShardId(), id.getLocalId(), timeMillis, value, stockpilePolicyId);
        } catch (NumberFormatException e) {
            throw MetricAddException.problem(MetricResponseStatus.REJECTED_MALFORMED);
        }
    }

    public String processName(String name) {
        if (!redirectHostMetrics) {
            return name;
        }
        String[] splits = name.split("\\.", 3);
        for (int i = 0; i < hostPostfixes.size(); i++) {
            if (splits[1].endsWith(hostPostfixes.get(i))) {
                return splits[0] + "." + hostMetricDir + "." + splits[1] + "." + splits[2];
            }
        }
        return name;
    }

    public boolean isReady() {
        return metricSearch.isInitialLoadComplete();
    }

    public void setRedirectHostMetrics(boolean redirectHostMetrics) {
        this.redirectHostMetrics = redirectHostMetrics;
    }
}
